"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InterceptsTriggerOrchestrator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _crypto = _interopRequireDefault(require("crypto"));
var _configSchema = require("@kbn/config-schema");
var _intervals = require("@kbn/task-manager-plugin/server/lib/intervals");
var _intercept_trigger = require("./services/intercept_trigger");
var _intercept_user_interaction = require("./services/intercept_user_interaction");
var _constants = require("../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class InterceptsTriggerOrchestrator {
  constructor() {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "usageCollector", void 0);
    (0, _defineProperty2.default)(this, "interceptTriggerService", new _intercept_trigger.InterceptTriggerService());
    (0, _defineProperty2.default)(this, "interceptUserInteractionService", new _intercept_user_interaction.InterceptUserInteractionService());
  }
  setup(core, {
    kibanaVersion,
    logger,
    usageCollection
  }) {
    this.logger = logger;
    this.usageCollector = this.setupUsageCollection(usageCollection);
    const {
      fetchRegisteredTask
    } = this.interceptTriggerService.setup(core, {
      kibanaVersion,
      usageCollector: this.usageCollector,
      logger: this.logger
    });
    this.interceptUserInteractionService.setup(core, this.logger);
    core.http.registerRouteHandlerContext('triggerInfo', this.routerHandlerContext.bind(this, fetchRegisteredTask));
    const router = core.http.createRouter();
    router.post.apply(router, this.getRouteConfig());
  }
  start(core) {
    const {
      registerTriggerDefinition
    } = this.interceptTriggerService.start(core);
    this.interceptUserInteractionService.start(core);
    return {
      registerTriggerDefinition
    };
  }

  /**
   * @description this method provides trigger information as context for the route handler
   */
  async routerHandlerContext(fetchRegisteredTask, ...args) {
    var _request$body;
    const [, request] = args;

    // @ts-expect-error -- the context is not typed
    const triggerId = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.triggerId;
    if (!triggerId) {
      return null;
    }
    let triggerInfo = null;
    let registeredTriggerDefinition;
    if (registeredTriggerDefinition = await fetchRegisteredTask(triggerId)) {
      triggerInfo = {
        registeredAt: registeredTriggerDefinition.firstRegisteredAt,
        triggerIntervalInMs: (0, _intervals.parseIntervalAsMillisecond)(registeredTriggerDefinition.triggerAfter),
        recurrent: registeredTriggerDefinition.recurrent
      };
    }
    return triggerInfo;
  }
  getRouteConfig() {
    return [{
      path: _constants.TRIGGER_INFO_API_ROUTE,
      validate: {
        body: _configSchema.schema.object({
          triggerId: _configSchema.schema.string()
        })
      },
      security: {
        authz: {
          enabled: false,
          reason: 'route is public and provides information about the intercept trigger'
        }
      }
    }, async (context, request, response) => {
      const resolvedTriggerInfo = await context.triggerInfo;
      if (!resolvedTriggerInfo) {
        return response.noContent();
      }

      // use the trigger interval as the etag
      const responseETag = _crypto.default.createHash('sha256').update(Buffer.from(String(resolvedTriggerInfo.triggerIntervalInMs))).digest('hex');
      if (request.headers['if-none-match'] === responseETag) {
        return response.notModified({});
      }
      return response.ok({
        headers: {
          etag: responseETag,
          // cache the response for the duration of the trigger interval
          'cache-control': `max-age=${resolvedTriggerInfo.triggerIntervalInMs}, must-revalidate`
        },
        body: resolvedTriggerInfo
      });
    }];
  }
  setupUsageCollection(usageCollection) {
    if (!usageCollection) {
      return;
    }
    return usageCollection === null || usageCollection === void 0 ? void 0 : usageCollection.createUsageCounter(_constants.USAGE_COLLECTION_DOMAIN_ID);
  }
}
exports.InterceptsTriggerOrchestrator = InterceptsTriggerOrchestrator;