"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useModuleStatus = void 0;
var _react = require("react");
var _log_analysis = require("../../../../common/log_analysis");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createInitialState = ({
  jobTypes
}) => ({
  jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
    ...accumulatedJobStatus,
    [jobType]: 'unknown'
  }), {}),
  jobSummaries: [],
  lastSetupErrorMessages: [],
  setupStatus: {
    type: 'initializing'
  }
});
const createStatusReducer = jobTypes => (state, action) => {
  switch (action.type) {
    case 'startedSetup':
      {
        return {
          ...state,
          jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
            ...accumulatedJobStatus,
            [jobType]: 'initializing'
          }), {}),
          setupStatus: {
            type: 'pending'
          }
        };
      }
    case 'finishedSetup':
      {
        const {
          datafeedSetupResults,
          jobSetupResults,
          jobSummaries,
          spaceId,
          logViewId,
          idFormat
        } = action;
        const nextJobStatus = jobTypes.reduce((accumulatedJobStatus, jobType) => ({
          ...accumulatedJobStatus,
          [jobType]: hasSuccessfullyCreatedJob((0, _log_analysis.getJobId)(spaceId, logViewId, idFormat, jobType))(jobSetupResults) && hasSuccessfullyStartedDatafeed((0, _log_analysis.getDatafeedId)(spaceId, logViewId, idFormat, jobType))(datafeedSetupResults) ? 'started' : 'failed'
        }), {});
        const nextSetupStatus = Object.values(nextJobStatus).every(jobState => jobState === 'started' || jobState === 'starting') ? {
          type: 'succeeded'
        } : {
          type: 'failed',
          reasons: [...Object.values(datafeedSetupResults).filter(hasError).map(datafeed => {
            var _datafeed$error$error;
            return (_datafeed$error$error = datafeed.error.error) === null || _datafeed$error$error === void 0 ? void 0 : _datafeed$error$error.reason;
          }), ...Object.values(jobSetupResults).filter(hasError).map(job => {
            var _job$error$error;
            return (_job$error$error = job.error.error) === null || _job$error$error === void 0 ? void 0 : _job$error$error.reason;
          })]
        };
        return {
          ...state,
          jobStatus: nextJobStatus,
          jobSummaries,
          setupStatus: nextSetupStatus
        };
      }
    case 'failedSetup':
      {
        return {
          ...state,
          jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
            ...accumulatedJobStatus,
            [jobType]: 'failed'
          }), {}),
          setupStatus: {
            type: 'failed',
            reasons: action.reason ? [action.reason] : ['unknown']
          }
        };
      }
    case 'fetchingJobStatuses':
      {
        return {
          ...state,
          setupStatus: state.setupStatus.type === 'unknown' ? {
            type: 'initializing'
          } : state.setupStatus
        };
      }
    case 'fetchedJobStatuses':
      {
        const {
          payload: jobSummaries,
          spaceId,
          logViewId,
          idFormat
        } = action;
        const {
          setupStatus
        } = state;
        const nextJobStatus = jobTypes.reduce((accumulatedJobStatus, jobType) => ({
          ...accumulatedJobStatus,
          [jobType]: getJobStatus((0, _log_analysis.getJobId)(spaceId, logViewId, idFormat, jobType))(jobSummaries)
        }), {});
        const nextSetupStatus = getSetupStatus(nextJobStatus)(setupStatus);
        return {
          ...state,
          jobSummaries,
          jobStatus: nextJobStatus,
          setupStatus: nextSetupStatus
        };
      }
    case 'failedFetchingJobStatuses':
      {
        return {
          ...state,
          setupStatus: {
            type: 'unknown'
          },
          jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
            ...accumulatedJobStatus,
            [jobType]: 'unknown'
          }), {})
        };
      }
    case 'viewedResults':
      {
        return {
          ...state,
          setupStatus: {
            type: 'skipped',
            newlyCreated: true
          }
        };
      }
    default:
      {
        return state;
      }
  }
};
const hasSuccessfullyCreatedJob = jobId => jobSetupResponses => jobSetupResponses.filter(jobSetupResponse => jobSetupResponse.id === jobId && jobSetupResponse.success && !jobSetupResponse.error).length > 0;
const hasSuccessfullyStartedDatafeed = datafeedId => datafeedSetupResponses => datafeedSetupResponses.filter(datafeedSetupResponse => datafeedSetupResponse.id === datafeedId && datafeedSetupResponse.success && datafeedSetupResponse.started && !datafeedSetupResponse.error).length > 0;
const getJobStatus = jobId => jobSummaries => jobSummaries.filter(jobSummary => jobSummary.id === jobId).map(jobSummary => {
  if (jobSummary.jobState === 'failed' || jobSummary.datafeedState === '') {
    return 'failed';
  } else if (jobSummary.jobState === 'closed' && jobSummary.datafeedState === 'stopped' && jobSummary.fullJob && jobSummary.fullJob.finished_time != null) {
    return 'finished';
  } else if (jobSummary.jobState === 'closed' || jobSummary.jobState === 'closing' || jobSummary.datafeedState === 'stopped') {
    return 'stopped';
  } else if (jobSummary.jobState === 'opening' && jobSummary.awaitingNodeAssignment === false || jobSummary.jobState === 'resetting' || jobSummary.jobState === 'reverting') {
    return 'initializing';
  } else if (jobSummary.jobState === 'opened' && jobSummary.datafeedState === 'started' || jobSummary.jobState === 'opening' && jobSummary.datafeedState === 'starting' && jobSummary.awaitingNodeAssignment === true) {
    return 'started';
  }
  return 'unknown';
})[0] || 'missing';
const getSetupStatus = everyJobStatus => previousSetupStatus => Object.entries(everyJobStatus).reduce((setupStatus, [, jobStatus]) => {
  if (jobStatus === 'missing') {
    return {
      type: 'required'
    };
  } else if (setupStatus.type === 'required' || setupStatus.type === 'succeeded') {
    return setupStatus;
  } else if (setupStatus.type === 'skipped' || (0, _log_analysis.isJobStatusWithResults)(jobStatus)) {
    return {
      type: 'skipped',
      // preserve newlyCreated status
      newlyCreated: setupStatus.type === 'skipped' && setupStatus.newlyCreated
    };
  }
  return setupStatus;
}, previousSetupStatus);
const hasError = value => value.error != null;
const useModuleStatus = jobTypes => {
  return (0, _react.useReducer)(createStatusReducer(jobTypes), {
    jobTypes
  }, createInitialState);
};
exports.useModuleStatus = useModuleStatus;