"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InferenceTaskErrorCode = exports.InferenceTaskError = void 0;
exports.createInferenceInternalError = createInferenceInternalError;
exports.createInferenceProviderError = createInferenceProviderError;
exports.createInferenceRequestAbortedError = createInferenceRequestAbortedError;
exports.createInferenceRequestError = createInferenceRequestError;
exports.isInferenceError = isInferenceError;
exports.isInferenceInternalError = isInferenceInternalError;
exports.isInferenceProviderError = isInferenceProviderError;
exports.isInferenceRequestAbortedError = isInferenceRequestAbortedError;
exports.isInferenceRequestError = isInferenceRequestError;
var _sseUtils = require("@kbn/sse-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * Enum for generic inference error codes.
 */
let InferenceTaskErrorCode = exports.InferenceTaskErrorCode = /*#__PURE__*/function (InferenceTaskErrorCode) {
  InferenceTaskErrorCode["providerError"] = "providerError";
  InferenceTaskErrorCode["internalError"] = "internalError";
  InferenceTaskErrorCode["requestError"] = "requestError";
  InferenceTaskErrorCode["abortedError"] = "requestAborted";
  return InferenceTaskErrorCode;
}({});
const InferenceTaskError = exports.InferenceTaskError = _sseUtils.ServerSentEventError;

/**
 * Inference error thrown when an unexpected internal error occurs while handling the request.
 */

/**
 * Inference error thrown when calling the provider through its connector returned an error.
 *
 * It includes error responses returned from the provider,
 * and any potential errors related to connectivity issue.
 */

/**
 * Inference error thrown when the request was considered invalid.
 *
 * Some example of reasons for invalid requests would be:
 * - no connector matching the provided connectorId
 * - invalid connector type for the provided connectorId
 */

/**
 * Inference error thrown when the request was aborted.
 *
 * Request abortion occurs when providing an abort signal and firing it
 * before the call to the LLM completes.
 */

function createInferenceInternalError(message = 'An internal error occurred', meta) {
  return new InferenceTaskError(InferenceTaskErrorCode.internalError, message, meta !== null && meta !== void 0 ? meta : {});
}
function createInferenceProviderError(message = 'An internal error occurred', meta) {
  return new InferenceTaskError(InferenceTaskErrorCode.providerError, message, meta !== null && meta !== void 0 ? meta : {});
}
function createInferenceRequestError(message, status) {
  return new InferenceTaskError(InferenceTaskErrorCode.requestError, message, {
    status
  });
}
function createInferenceRequestAbortedError() {
  return new InferenceTaskError(InferenceTaskErrorCode.abortedError, 'Request was aborted', {
    status: 499
  });
}

/**
 * Check if the given error is an {@link InferenceTaskError}
 */
function isInferenceError(error) {
  return error instanceof InferenceTaskError;
}

/**
 * Check if the given error is an {@link InferenceTaskInternalError}
 */
function isInferenceInternalError(error) {
  return isInferenceError(error) && error.code === InferenceTaskErrorCode.internalError;
}

/**
 * Check if the given error is an {@link InferenceTaskRequestError}
 */
function isInferenceRequestError(error) {
  return isInferenceError(error) && error.code === InferenceTaskErrorCode.requestError;
}

/**
 * Check if the given error is an {@link InferenceTaskAbortedError}
 */
function isInferenceRequestAbortedError(error) {
  return isInferenceError(error) && error.code === InferenceTaskErrorCode.abortedError;
}

/**
 * Check if the given error is an {@link InferenceTaskProviderError}
 */
function isInferenceProviderError(error) {
  return isInferenceError(error) && error.code === InferenceTaskErrorCode.providerError;
}