"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGetAllRoute = registerGetAllRoute;
exports.registerGetOneRoute = registerGetOneRoute;
var _configSchema = require("@kbn/config-schema");
var _data_stream_serialization = require("../../../lib/data_stream_serialization");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const enhanceDataStreams = ({
  dataStreams,
  dataStreamsStats,
  meteringStats,
  dataStreamsPrivileges,
  globalMaxRetention
}) => {
  return dataStreams.map(dataStream => {
    const enhancedDataStream = {
      ...dataStream,
      ...(globalMaxRetention ? {
        global_max_retention: globalMaxRetention
      } : {}),
      privileges: {
        delete_index: dataStreamsPrivileges ? dataStreamsPrivileges.index[dataStream.name].delete_index : true,
        manage_data_stream_lifecycle: dataStreamsPrivileges ? dataStreamsPrivileges.index[dataStream.name].manage_data_stream_lifecycle : true,
        read_failure_store: dataStreamsPrivileges ? dataStreamsPrivileges.index[dataStream.name].read_failure_store : true
      }
    };
    if (dataStreamsStats) {
      const currentDataStreamStats = dataStreamsStats.find(({
        data_stream: statsName
      }) => statsName === dataStream.name);
      if (currentDataStreamStats) {
        enhancedDataStream.store_size = currentDataStreamStats.store_size;
        enhancedDataStream.store_size_bytes = currentDataStreamStats.store_size_bytes;
        enhancedDataStream.maximum_timestamp = currentDataStreamStats.maximum_timestamp;
      }
    }
    if (meteringStats) {
      const datastreamMeteringStats = meteringStats.find(s => s.name === dataStream.name);
      if (datastreamMeteringStats) {
        enhancedDataStream.metering_size_in_bytes = datastreamMeteringStats.size_in_bytes;
        enhancedDataStream.metering_doc_count = datastreamMeteringStats.num_docs;
      }
    }
    return enhancedDataStream;
  });
};
const getDataStreams = (client, name = '*') => {
  return client.asCurrentUser.indices.getDataStream({
    name,
    expand_wildcards: 'all'
  });
};
const getDataStreamLifecycle = (client, name) => {
  return client.asCurrentUser.indices.getDataLifecycle({
    name
  });
};
const getDataStreamsStats = (client, name = '*') => {
  return client.asCurrentUser.indices.dataStreamsStats({
    name,
    expand_wildcards: 'all',
    human: true
  });
};
const getMeteringStats = (client, name) => {
  let path = `/_metering/stats`;
  if (name) {
    path = `${path}/${name}`;
  }
  return client.asSecondaryAuthUser.transport.request({
    method: 'GET',
    path
  });
};
const getDataStreamsPrivileges = (client, names) => {
  return client.asCurrentUser.security.hasPrivileges({
    index: [{
      names,
      privileges: ['delete_index', 'manage_data_stream_lifecycle', 'read_failure_store']
    }]
  });
};
function registerGetAllRoute({
  router,
  lib: {
    handleEsError
  },
  config
}) {
  const querySchema = _configSchema.schema.object({
    includeStats: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('true'), _configSchema.schema.literal('false')]))
  });
  router.get({
    path: (0, _.addBasePath)('/data_streams'),
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      query: querySchema
    }
  }, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const includeStats = request.query.includeStats === 'true';
    try {
      var _persistent$cluster$l, _persistent$cluster, _persistent$cluster$l2, _defaults$cluster, _defaults$cluster$log, _persistent$data_stre, _persistent$data_stre2, _persistent$data_stre3, _defaults$data_stream, _defaults$data_stream2, _lifecycle$global_ret;
      const {
        data_streams: dataStreams
      } = await getDataStreams(client);
      let dataStreamsStats;
      let dataStreamsPrivileges;
      let meteringStats;
      if (includeStats && config.isDataStreamStatsEnabled !== false) {
        ({
          data_streams: dataStreamsStats
        } = await getDataStreamsStats(client));
      }
      if (includeStats && config.isSizeAndDocCountEnabled !== false) {
        ({
          datastreams: meteringStats
        } = await getMeteringStats(client));
      }
      if (config.isSecurityEnabled() && dataStreams.length > 0) {
        dataStreamsPrivileges = await getDataStreamsPrivileges(client, dataStreams.map(dataStream => dataStream.name));
      }
      const {
        persistent,
        defaults
      } = await client.asInternalUser.cluster.getSettings({
        include_defaults: true
      });
      const isLogsdbEnabled = ((_persistent$cluster$l = persistent === null || persistent === void 0 ? void 0 : (_persistent$cluster = persistent.cluster) === null || _persistent$cluster === void 0 ? void 0 : (_persistent$cluster$l2 = _persistent$cluster.logsdb) === null || _persistent$cluster$l2 === void 0 ? void 0 : _persistent$cluster$l2.enabled) !== null && _persistent$cluster$l !== void 0 ? _persistent$cluster$l : defaults === null || defaults === void 0 ? void 0 : (_defaults$cluster = defaults.cluster) === null || _defaults$cluster === void 0 ? void 0 : (_defaults$cluster$log = _defaults$cluster.logsdb) === null || _defaults$cluster$log === void 0 ? void 0 : _defaults$cluster$log.enabled) === 'true';

      // Get failure store cluster settings
      const failureStoreSettings = {
        enabled: (_persistent$data_stre = persistent === null || persistent === void 0 ? void 0 : (_persistent$data_stre2 = persistent.data_streams) === null || _persistent$data_stre2 === void 0 ? void 0 : (_persistent$data_stre3 = _persistent$data_stre2.failure_store) === null || _persistent$data_stre3 === void 0 ? void 0 : _persistent$data_stre3.enabled) !== null && _persistent$data_stre !== void 0 ? _persistent$data_stre : defaults === null || defaults === void 0 ? void 0 : (_defaults$data_stream = defaults.data_streams) === null || _defaults$data_stream === void 0 ? void 0 : (_defaults$data_stream2 = _defaults$data_stream.failure_store) === null || _defaults$data_stream2 === void 0 ? void 0 : _defaults$data_stream2.enabled
      };

      // Only take the lifecycle of the first data stream since all data streams have the same global retention period
      const lifecycle = await getDataStreamLifecycle(client, dataStreams[0].name);
      // @ts-ignore - TS doesn't know about the `global_retention` property yet
      const globalMaxRetention = lifecycle === null || lifecycle === void 0 ? void 0 : (_lifecycle$global_ret = lifecycle.global_retention) === null || _lifecycle$global_ret === void 0 ? void 0 : _lifecycle$global_ret.max_retention;
      const enhancedDataStreams = enhanceDataStreams({
        dataStreams,
        dataStreamsStats,
        meteringStats,
        dataStreamsPrivileges,
        globalMaxRetention
      });
      return response.ok({
        body: (0, _data_stream_serialization.deserializeDataStreamList)(enhancedDataStreams, isLogsdbEnabled, failureStoreSettings)
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}
function registerGetOneRoute({
  router,
  lib: {
    handleEsError
  },
  config
}) {
  const paramsSchema = _configSchema.schema.object({
    name: _configSchema.schema.string()
  });
  router.get({
    path: (0, _.addBasePath)('/data_streams/{name}'),
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      params: paramsSchema
    }
  }, async (context, request, response) => {
    const {
      name
    } = request.params;
    const {
      client
    } = (await context.core).elasticsearch;
    let dataStreamsStats;
    let meteringStats;
    try {
      var _lifecycle$global_ret2;
      const {
        data_streams: dataStreams
      } = await getDataStreams(client, name);
      const lifecycle = await getDataStreamLifecycle(client, name);
      // @ts-ignore - TS doesn't know about the `global_retention` property yet
      const globalMaxRetention = lifecycle === null || lifecycle === void 0 ? void 0 : (_lifecycle$global_ret2 = lifecycle.global_retention) === null || _lifecycle$global_ret2 === void 0 ? void 0 : _lifecycle$global_ret2.max_retention;
      if (config.isDataStreamStatsEnabled !== false) {
        ({
          data_streams: dataStreamsStats
        } = await getDataStreamsStats(client, name));
      }
      if (config.isSizeAndDocCountEnabled !== false) {
        ({
          datastreams: meteringStats
        } = await getMeteringStats(client, name));
      }
      if (dataStreams[0]) {
        var _persistent$data_stre4, _persistent$data_stre5, _persistent$data_stre6, _defaults$data_stream3, _defaults$data_stream4, _persistent$cluster$l3, _persistent$cluster2, _persistent$cluster2$, _defaults$cluster2, _defaults$cluster2$lo;
        let dataStreamsPrivileges;
        if (config.isSecurityEnabled()) {
          dataStreamsPrivileges = await getDataStreamsPrivileges(client, [dataStreams[0].name]);
        }
        const {
          persistent,
          defaults
        } = await client.asInternalUser.cluster.getSettings({
          include_defaults: true
        });

        // Get failure store cluster settings
        const failureStoreSettings = {
          enabled: (_persistent$data_stre4 = persistent === null || persistent === void 0 ? void 0 : (_persistent$data_stre5 = persistent.data_streams) === null || _persistent$data_stre5 === void 0 ? void 0 : (_persistent$data_stre6 = _persistent$data_stre5.failure_store) === null || _persistent$data_stre6 === void 0 ? void 0 : _persistent$data_stre6.enabled) !== null && _persistent$data_stre4 !== void 0 ? _persistent$data_stre4 : defaults === null || defaults === void 0 ? void 0 : (_defaults$data_stream3 = defaults.data_streams) === null || _defaults$data_stream3 === void 0 ? void 0 : (_defaults$data_stream4 = _defaults$data_stream3.failure_store) === null || _defaults$data_stream4 === void 0 ? void 0 : _defaults$data_stream4.enabled
        };
        const enhancedDataStreams = enhanceDataStreams({
          dataStreams,
          dataStreamsStats,
          meteringStats,
          dataStreamsPrivileges,
          globalMaxRetention
        });
        const isLogsdbEnabled = ((_persistent$cluster$l3 = persistent === null || persistent === void 0 ? void 0 : (_persistent$cluster2 = persistent.cluster) === null || _persistent$cluster2 === void 0 ? void 0 : (_persistent$cluster2$ = _persistent$cluster2.logsdb) === null || _persistent$cluster2$ === void 0 ? void 0 : _persistent$cluster2$.enabled) !== null && _persistent$cluster$l3 !== void 0 ? _persistent$cluster$l3 : defaults === null || defaults === void 0 ? void 0 : (_defaults$cluster2 = defaults.cluster) === null || _defaults$cluster2 === void 0 ? void 0 : (_defaults$cluster2$lo = _defaults$cluster2.logsdb) === null || _defaults$cluster2$lo === void 0 ? void 0 : _defaults$cluster2$lo.enabled) === 'true';
        const body = (0, _data_stream_serialization.deserializeDataStream)(enhancedDataStreams[0], isLogsdbEnabled, failureStoreSettings);
        return response.ok({
          body
        });
      }
      return response.notFound();
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
}