"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useConvertIndexToLookup = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _upgradeAssistantPkgCommon = require("@kbn/upgrade-assistant-pkg-common");
var _services = require("../../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const POLL_INTERVAL = 3000;
const useConvertIndexToLookup = ({
  sourceIndexName,
  onSuccess,
  onClose
}) => {
  const [isConverting, setIsConverting] = (0, _react.useState)(false);
  const [errorMessage, setErrorMessage] = (0, _react.useState)('');
  const onSuccessRef = (0, _react.useRef)(onSuccess);
  const onCloseRef = (0, _react.useRef)(onClose);
  (0, _react.useEffect)(() => {
    onSuccessRef.current = onSuccess;
    onCloseRef.current = onClose;
  }, [onSuccess, onClose]);
  const pollIntervalIdRef = (0, _react.useRef)(null);
  const clearPollInterval = (0, _react.useCallback)(() => {
    if (pollIntervalIdRef.current) {
      clearTimeout(pollIntervalIdRef.current);
      pollIntervalIdRef.current = null;
    }
  }, []);
  const updateStatus = (0, _react.useCallback)(async () => {
    clearPollInterval();
    try {
      var _data$reindexOp, _data$reindexOp2, _data$reindexOp3;
      const {
        data,
        error
      } = await (0, _services.getReindexStatus)(sourceIndexName);
      if (error) {
        setErrorMessage(error.message);
        setIsConverting(false);
        return;
      }
      if ((data === null || data === void 0 ? void 0 : (_data$reindexOp = data.reindexOp) === null || _data$reindexOp === void 0 ? void 0 : _data$reindexOp.status) === _upgradeAssistantPkgCommon.ReindexStatus.inProgress) {
        pollIntervalIdRef.current = window.setTimeout(updateStatus, POLL_INTERVAL);
      } else if ((data === null || data === void 0 ? void 0 : (_data$reindexOp2 = data.reindexOp) === null || _data$reindexOp2 === void 0 ? void 0 : _data$reindexOp2.status) === _upgradeAssistantPkgCommon.ReindexStatus.failed || (data === null || data === void 0 ? void 0 : (_data$reindexOp3 = data.reindexOp) === null || _data$reindexOp3 === void 0 ? void 0 : _data$reindexOp3.status) === _upgradeAssistantPkgCommon.ReindexStatus.cancelled) {
        var _data$reindexOp$error, _data$reindexOp4;
        setErrorMessage((_data$reindexOp$error = data === null || data === void 0 ? void 0 : (_data$reindexOp4 = data.reindexOp) === null || _data$reindexOp4 === void 0 ? void 0 : _data$reindexOp4.errorMessage) !== null && _data$reindexOp$error !== void 0 ? _data$reindexOp$error : _i18n.i18n.translate('xpack.idxMgmt.convertToLookupIndexAction.reindexFailedOrCancelledErrorMessage', {
          defaultMessage: 'Reindex did not complete successfully due to unknown error.'
        }));
        setIsConverting(false);
        return;
      } else {
        var _data$reindexOp5;
        setIsConverting(false);
        onCloseRef.current();
        if (data !== null && data !== void 0 && (_data$reindexOp5 = data.reindexOp) !== null && _data$reindexOp5 !== void 0 && _data$reindexOp5.newIndexName) {
          onSuccessRef.current(data.reindexOp.newIndexName);
        }
      }
    } catch {
      setErrorMessage(_i18n.i18n.translate('xpack.idxMgmt.convertToLookupIndexAction.unexpectedErrorMessage', {
        defaultMessage: 'An unexpected error occurred.'
      }));
      setIsConverting(false);
    }
  }, [clearPollInterval, sourceIndexName]);
  const convert = async lookupIndexName => {
    setIsConverting(true);
    setErrorMessage('');
    try {
      const {
        error
      } = await (0, _services.startReindex)(sourceIndexName, lookupIndexName);
      if (error) {
        setErrorMessage(error.message);
        setIsConverting(false);
        return;
      }
      await updateStatus();
    } catch {
      setErrorMessage(_i18n.i18n.translate('xpack.idxMgmt.convertToLookupIndexAction.unexpectedErrorMessage', {
        defaultMessage: 'An unexpected error occurred.'
      }));
      setIsConverting(false);
    }
  };
  (0, _react.useEffect)(() => {
    return () => {
      clearPollInterval();
    };
  }, [clearPollInterval, sourceIndexName]);
  return {
    isConverting,
    errorMessage,
    convert
  };
};
exports.useConvertIndexToLookup = useConvertIndexToLookup;