"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndexAdapter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _create_or_update_component_template = require("./create_or_update_component_template");
var _create_or_update_index = require("./create_or_update_index");
var _create_or_update_index_template = require("./create_or_update_index_template");
var _install_with_timeout = require("./install_with_timeout");
var _resource_installer_utils = require("./resource_installer_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_FIELDS_LIMIT = 2500;
class IndexAdapter {
  constructor(name, options) {
    var _options$totalFieldsL, _options$writeIndexOn;
    (0, _defineProperty2.default)(this, "kibanaVersion", void 0);
    (0, _defineProperty2.default)(this, "totalFieldsLimit", void 0);
    (0, _defineProperty2.default)(this, "componentTemplates", []);
    (0, _defineProperty2.default)(this, "indexTemplates", []);
    (0, _defineProperty2.default)(this, "installed", void 0);
    (0, _defineProperty2.default)(this, "writeIndexOnly", void 0);
    this.name = name;
    this.installed = false;
    this.kibanaVersion = options.kibanaVersion;
    this.totalFieldsLimit = (_options$totalFieldsL = options.totalFieldsLimit) !== null && _options$totalFieldsL !== void 0 ? _options$totalFieldsL : DEFAULT_FIELDS_LIMIT;
    this.writeIndexOnly = (_options$writeIndexOn = options.writeIndexOnly) !== null && _options$writeIndexOn !== void 0 ? _options$writeIndexOn : false;
  }
  setComponentTemplate(params) {
    if (this.installed) {
      throw new Error('Cannot set component template after install');
    }
    this.componentTemplates.push((0, _resource_installer_utils.getComponentTemplate)(params));
  }
  setIndexTemplate(params) {
    if (this.installed) {
      throw new Error('Cannot set index template after install');
    }
    this.indexTemplates.push((0, _resource_installer_utils.getIndexTemplate)({
      ...params,
      indexPatterns: [this.name],
      kibanaVersion: this.kibanaVersion,
      totalFieldsLimit: this.totalFieldsLimit
    }));
  }
  getInstallFn({
    logger,
    pluginStop$,
    tasksTimeoutMs
  }) {
    return async (promise, description) => {
      try {
        await (0, _install_with_timeout.installWithTimeout)({
          installFn: () => promise,
          description,
          timeoutMs: tasksTimeoutMs,
          pluginStop$
        });
      } catch (err) {
        if (err instanceof _install_with_timeout.InstallShutdownError) {
          logger.info(err.message);
        } else {
          throw err;
        }
      }
    };
  }
  async installTemplates(params) {
    const {
      logger,
      pluginStop$,
      tasksTimeoutMs
    } = params;
    const esClient = await params.esClient;
    const installFn = this.getInstallFn({
      logger,
      pluginStop$,
      tasksTimeoutMs
    });

    // Install component templates in parallel
    await Promise.all(this.componentTemplates.map(componentTemplate => installFn((0, _create_or_update_component_template.createOrUpdateComponentTemplate)({
      template: componentTemplate,
      esClient,
      logger,
      totalFieldsLimit: this.totalFieldsLimit
    }), `create or update ${componentTemplate.name} component template`)));

    // Install index templates in parallel
    await Promise.all(this.indexTemplates.map(indexTemplate => installFn((0, _create_or_update_index_template.createOrUpdateIndexTemplate)({
      template: indexTemplate,
      esClient,
      logger
    }), `create or update ${indexTemplate.name} index template`)));
  }
  async install(params) {
    this.installed = true;
    const {
      logger,
      pluginStop$,
      tasksTimeoutMs
    } = params;
    const esClient = await params.esClient;
    await this.installTemplates(params);
    const installFn = this.getInstallFn({
      logger,
      pluginStop$,
      tasksTimeoutMs
    });

    // create index when everything is ready
    await installFn((0, _create_or_update_index.createOrUpdateIndex)({
      name: this.name,
      esClient,
      logger,
      totalFieldsLimit: this.totalFieldsLimit
    }), `${this.name} index`);
  }
}
exports.IndexAdapter = IndexAdapter;