"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseSearchParams = void 0;
var _eui = require("@elastic/eui");
var _query_utils = require("./query_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const knownFilters = ['tag', 'type'];
const aliasMap = {
  tag: ['tags'],
  type: ['types']
};

// Converts multiword types to phrases by wrapping them in quotes and trimming whitespace after type keyword. Example: type:  canvas workpad -> type:"canvas workpad". If the type is already wrapped in quotes or is a single word, it will only trim whitespace after type keyword.
const convertMultiwordTypesToPhrasesAndTrimWhitespace = (term, multiWordTypes) => {
  if (!multiWordTypes.length) {
    return term.replace(/(type:|types:)\s*([^"']*?)\b([^"'\s]+)/gi, (_, typeKeyword, whitespace, typeValue) => `${typeKeyword}${whitespace.trim()}${typeValue}`);
  }
  const typesPattern = multiWordTypes.join('|');
  const termReplaceRegex = new RegExp(`(type:|types:)\\s*([^"']*?)\\b((${typesPattern})\\b|[^\\s"']+)`, 'gi');
  return term.replace(termReplaceRegex, (_, typeKeyword, whitespace, typeValue) => {
    const trimmedTypeKeyword = `${typeKeyword}${whitespace.trim()}`;

    // If the type value is already wrapped in quotes, leave it as is
    return /['"]/.test(typeValue) ? `${trimmedTypeKeyword}${typeValue}` : `${trimmedTypeKeyword}"${typeValue}"`;
  });
};
const dedupeTypes = types => [...new Set(types.map(item => item.replace(/[-\s]+/g, ' ').trim()))];
const parseSearchParams = (term, searchableTypes) => {
  const recognizedFields = knownFilters.concat(...Object.values(aliasMap));
  let query;

  // Finds all multiword types that are separated by whitespace or hyphens
  const multiWordSearchableTypesWhitespaceSeperated = searchableTypes.filter(item => /[ -]/.test(item)).map(item => item.replace(/-/g, ' '));
  const modifiedTerm = convertMultiwordTypesToPhrasesAndTrimWhitespace(term, multiWordSearchableTypesWhitespaceSeperated);
  try {
    query = _eui.Query.parse(modifiedTerm, {
      schema: {
        recognizedFields
      }
    });
  } catch (e) {
    // if the query fails to parse, we just perform the search against the raw search term.
    return {
      term,
      filters: {}
    };
  }
  const searchTerm = (0, _query_utils.getSearchTerm)(query);
  const filterValues = (0, _query_utils.applyAliases)((0, _query_utils.getFieldValueMap)(query), aliasMap);
  const tags = filterValues.get('tag');
  const types = filterValues.get('type');
  return {
    term: searchTerm,
    filters: {
      tags: tags ? valuesToString(tags) : undefined,
      types: types ? dedupeTypes(valuesToString(types)) : undefined
    }
  };
};
exports.parseSearchParams = parseSearchParams;
const valuesToString = raw => raw.map(value => String(value));