"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.canEnableSyncIntegrations = exports.FLEET_SYNCED_INTEGRATIONS_INDEX_NAME = exports.FLEET_SYNCED_INTEGRATIONS_INDEX_CONFIG = exports.FLEET_SYNCED_INTEGRATIONS_CCR_INDEX_PREFIX = void 0;
exports.createCCSIndexPatterns = createCCSIndexPatterns;
exports.createOrUpdateFleetSyncedIntegrationsIndex = createOrUpdateFleetSyncedIntegrationsIndex;
var _utils = require("@kbn/utils");
var _errors = require("../../errors");
var _app_context = require("../app_context");
var _install = require("../epm/kibana/index_pattern/install");
var _constants = require("../../constants");
var _license = require("../license");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FLEET_SYNCED_INTEGRATIONS_INDEX_NAME = exports.FLEET_SYNCED_INTEGRATIONS_INDEX_NAME = 'fleet-synced-integrations';
const FLEET_SYNCED_INTEGRATIONS_CCR_INDEX_PREFIX = exports.FLEET_SYNCED_INTEGRATIONS_CCR_INDEX_PREFIX = 'fleet-synced-integrations-ccr-*';
const FLEET_SYNCED_INTEGRATIONS_INDEX_CONFIG = exports.FLEET_SYNCED_INTEGRATIONS_INDEX_CONFIG = {
  settings: {
    auto_expand_replicas: '0-1'
  },
  mappings: {
    dynamic: false,
    _meta: {
      version: '1.0'
    },
    properties: {
      remote_es_hosts: {
        properties: {
          name: {
            type: 'keyword'
          },
          hosts: {
            type: 'keyword'
          },
          sync_integrations: {
            type: 'boolean'
          }
        }
      },
      integrations: {
        properties: {
          package_name: {
            type: 'keyword'
          },
          package_version: {
            type: 'keyword'
          },
          updated_at: {
            type: 'date'
          }
        }
      }
    }
  }
};
const canEnableSyncIntegrations = () => {
  var _appContextService$ge;
  const {
    enableSyncIntegrationsOnRemote
  } = _app_context.appContextService.getExperimentalFeatures();
  const isServerless = (_appContextService$ge = _app_context.appContextService.getCloud()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.isServerlessEnabled;
  return enableSyncIntegrationsOnRemote && _license.licenseService.isEnterprise() && !isServerless;
};
exports.canEnableSyncIntegrations = canEnableSyncIntegrations;
async function createOrUpdateFleetSyncedIntegrationsIndex(esClient) {
  _app_context.appContextService.getLogger().debug('Create or update fleet-synced-integrations index');
  await createOrUpdateIndex(esClient, FLEET_SYNCED_INTEGRATIONS_INDEX_NAME, FLEET_SYNCED_INTEGRATIONS_INDEX_CONFIG);
}
async function createOrUpdateIndex(esClient, indexName, indexData) {
  const resExists = await esClient.indices.exists({
    index: indexName
  });
  if (resExists) {
    return updateIndex(esClient, indexName, indexData);
  }
  return createIndex(esClient, indexName, indexData);
}
async function updateIndex(esClient, indexName, indexData) {
  try {
    var _res$indexName$mappin, _res$indexName$mappin2;
    const res = await esClient.indices.getMapping({
      index: indexName
    });
    const versionChanged = ((_res$indexName$mappin = res[indexName].mappings) === null || _res$indexName$mappin === void 0 ? void 0 : (_res$indexName$mappin2 = _res$indexName$mappin._meta) === null || _res$indexName$mappin2 === void 0 ? void 0 : _res$indexName$mappin2.version) !== indexData.mappings._meta.version;
    if (versionChanged) {
      await esClient.indices.putMapping({
        index: indexName,
        body: Object.assign({
          ...indexData.mappings
        })
      });
    }
  } catch (err) {
    throw new _errors.FleetSetupError(`update of index [${indexName}] failed ${err}`);
  }
}
async function createIndex(esClient, indexName, indexData) {
  try {
    await esClient.indices.create({
      index: indexName,
      body: indexData
    });
  } catch (err) {
    var _err$body, _err$body$error;
    if ((err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) !== 'resource_already_exists_exception') {
      throw new _errors.FleetSetupError(`create of index [${indexName}] failed ${err}`);
    }
  }
}
async function createCCSIndexPatterns(esClient, savedObjectsClient, savedObjectsImporter) {
  var _appContextService$ge2;
  if (!canEnableSyncIntegrations()) {
    return;
  }
  if (((_appContextService$ge2 = _app_context.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.enableManagedLogsAndMetricsDataviews) !== true) {
    return;
  }
  let remoteClusterNames = [];
  try {
    const remoteInfo = await esClient.cluster.remoteInfo();
    remoteClusterNames = Object.keys(remoteInfo);
  } catch (error) {
    _app_context.appContextService.getLogger().warn(`Error fetching remote cluster info: ${error.message}`);
    return;
  }
  if (remoteClusterNames.length === 0) {
    return;
  }
  const indexPatternSavedObjectsWithRemoteCluster = [];
  for (const clusterName of remoteClusterNames) {
    for (const indexPatternType of _install.indexPatternTypes) {
      indexPatternSavedObjectsWithRemoteCluster.push({
        id: `${clusterName}:${indexPatternType}-*`,
        type: _install.INDEX_PATTERN_SAVED_OBJECT_TYPE,
        typeMigrationVersion: '8.0.0',
        attributes: {
          title: `${clusterName}:${indexPatternType}-*`,
          timeFieldName: '@timestamp',
          allowNoIndex: true
        },
        references: []
      });
    }
  }
  const results = await savedObjectsClient.find({
    type: _install.INDEX_PATTERN_SAVED_OBJECT_TYPE,
    perPage: _constants.SO_SEARCH_LIMIT,
    namespaces: ['*'],
    fields: ['namespaces']
  });
  const existingIndexPatterns = results.saved_objects.reduce((acc, savedObject) => {
    var _savedObject$namespac;
    acc[savedObject.id] = {
      namespaces: (_savedObject$namespac = savedObject.namespaces) !== null && _savedObject$namespac !== void 0 ? _savedObject$namespac : [],
      id: savedObject.id
    };
    return acc;
  }, {});
  const notExistingIndexPatterns = indexPatternSavedObjectsWithRemoteCluster.filter(indexPatternSavedObject => !existingIndexPatterns[indexPatternSavedObject.id]);
  if (notExistingIndexPatterns.length > 0) {
    await savedObjectsImporter.import({
      overwrite: false,
      readStream: (0, _utils.createListStream)(notExistingIndexPatterns),
      createNewCopies: false,
      refresh: false,
      managed: true
    });
  }
  const indexPatternsNotInAllSpaces = indexPatternSavedObjectsWithRemoteCluster.filter(indexPatternSavedObject => !existingIndexPatterns[indexPatternSavedObject.id] || !existingIndexPatterns[indexPatternSavedObject.id].namespaces.includes('*'));

  // Make index patterns available in all spaces
  for (const indexPatternSavedObject of indexPatternsNotInAllSpaces) {
    try {
      await savedObjectsClient.updateObjectsSpaces([{
        id: indexPatternSavedObject.id,
        type: _install.INDEX_PATTERN_SAVED_OBJECT_TYPE
      }], ['*'], []);
    } catch (error) {
      _app_context.appContextService.getLogger().error(`Error making managed index patterns global: ${error.message}`);
    }
  }
}