"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSecrets = createSecrets;
exports.deleteSOSecrets = deleteSOSecrets;
exports.deleteSecrets = deleteSecrets;
exports.diffSOSecretPaths = diffSOSecretPaths;
exports.extractAndUpdateSOSecrets = extractAndUpdateSOSecrets;
exports.extractAndWriteSOSecrets = extractAndWriteSOSecrets;
exports.isSecretStorageEnabled = isSecretStorageEnabled;
exports.toCompiledSecretRef = toCompiledSecretRef;
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _errors = require("../../errors");
var _constants = require("../../constants");
var _retry = require("../epm/elasticsearch/retry");
var _audit_logging = require("../audit_logging");
var _app_context = require("../app_context");
var _ = require("..");
var _fleet_server = require("../fleet_server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function isSecretStorageEnabled(esClient, soClient) {
  var _appContextService$ge, _appContextService$ge2, _appContextService$ge3;
  const logger = _app_context.appContextService.getLogger();

  // if serverless then secrets will always be supported
  const isFleetServerStandalone = (_appContextService$ge = (_appContextService$ge2 = _app_context.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : (_appContextService$ge3 = _appContextService$ge2.internal) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.fleetServerStandalone) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : false;
  if (isFleetServerStandalone) {
    logger.trace('Secrets storage is enabled as fleet server is standalone');
    return true;
  }

  // now check the flag in settings to see if the fleet server requirement has already been met
  // once the requirement has been met, secrets are always on
  const settings = await _.settingsService.getSettingsOrUndefined(soClient);
  if (settings && settings.secret_storage_requirements_met) {
    logger.debug('Secrets storage requirements already met, turned on in settings');
    return true;
  }
  const areAllFleetServersOnProperVersion = await (0, _fleet_server.checkFleetServerVersionsForSecretsStorage)(esClient, soClient, _constants.SECRETS_MINIMUM_FLEET_SERVER_VERSION);

  // otherwise check if we have the minimum fleet server version and enable secrets if so
  if (areAllFleetServersOnProperVersion) {
    logger.debug('Enabling secrets storage as minimum fleet server version has been met');
    try {
      await _.settingsService.saveSettings(soClient, {
        secret_storage_requirements_met: true
      });
    } catch (err) {
      // we can suppress this error as it will be retried on the next function call
      logger.warn(`Failed to save settings after enabling secrets storage: ${err.message}`);
    }
    return true;
  }
  logger.info('Secrets storage is disabled as minimum fleet server version has not been met');
  return false;
}
async function createSecrets(opts) {
  const {
    esClient,
    values
  } = opts;
  const logger = _app_context.appContextService.getLogger();
  const sendESRequest = value => {
    return (0, _retry.retryTransientEsErrors)(() => esClient.transport.request({
      method: 'POST',
      path: _constants.SECRETS_ENDPOINT_PATH,
      body: {
        value
      }
    }), {
      logger
    });
  };
  const secretsResponse = await Promise.all(values.map(async value => {
    try {
      if (Array.isArray(value)) {
        return await Promise.all(value.map(sendESRequest));
      } else {
        return await sendESRequest(value);
      }
    } catch (err) {
      const msg = `Error creating secrets: ${err}`;
      logger.error(msg);
      throw new _errors.FleetError(msg);
    }
  }));
  const writeLog = item => {
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `secret created: ${item.id}`,
      event: {
        action: 'secret_create',
        category: ['database'],
        type: ['access'],
        outcome: 'success'
      }
    });
  };
  secretsResponse.forEach(item => {
    if (Array.isArray(item)) {
      item.forEach(writeLog);
    } else {
      writeLog(item);
    }
  });
  return secretsResponse;
}
async function deleteSecrets(opts) {
  const {
    esClient,
    ids
  } = opts;
  const logger = _app_context.appContextService.getLogger();
  const deletedRes = await Promise.all(ids.map(async id => {
    try {
      const getDeleteRes = await (0, _retry.retryTransientEsErrors)(() => esClient.transport.request({
        method: 'DELETE',
        path: `${_constants.SECRETS_ENDPOINT_PATH}/${id}`
      }), {
        logger
      });
      return {
        ...getDeleteRes,
        id
      };
    } catch (err) {
      const msg = `Error deleting secrets: ${err}`;
      logger.error(msg);
      throw new _errors.FleetError(msg);
    }
  }));
  deletedRes.forEach(item => {
    if (item.deleted === true) {
      _audit_logging.auditLoggingService.writeCustomAuditLog({
        message: `secret deleted: ${item.id}`,
        event: {
          action: 'secret_delete',
          category: ['database'],
          type: ['access'],
          outcome: 'success'
        }
      });
    }
  });
}

// this is how IDs are inserted into compiled templates
function toCompiledSecretRef(id) {
  return `$co.elastic.secret{${id}}`;
}

/**
 * Given an array of secret paths, deletes the corresponding secrets
 */
async function deleteSOSecrets(esClient, secretPaths) {
  if (secretPaths.length === 0) {
    return Promise.resolve();
  }
  const secretIds = secretPaths.map(({
    value
  }) => value.id);
  try {
    return deleteSecrets({
      esClient,
      ids: secretIds
    });
  } catch (err) {
    _app_context.appContextService.getLogger().warn(`Error deleting secrets: ${err}`);
  }
}

/**
 * Takes a generic object T and its secret paths
 * Creates new secrets and returns the references
 */
async function extractAndWriteSOSecrets(opts) {
  const {
    soObject,
    esClient,
    secretPaths,
    secretHashes = {}
  } = opts;
  if (secretPaths.length === 0) {
    return {
      soObjectWithSecrets: soObject,
      secretReferences: []
    };
  }
  const secrets = await createSecrets({
    esClient,
    values: secretPaths.map(({
      value
    }) => value)
  });
  const objectWithSecretRefs = JSON.parse(JSON.stringify(soObject));
  secretPaths.forEach((secretPath, i) => {
    const pathWithoutPrefix = secretPath.path.replace('secrets.', '');
    const maybeHash = (0, _lodash.get)(secretHashes, pathWithoutPrefix);
    const currentSecret = secrets[i];
    (0, _saferLodashSet.set)(objectWithSecretRefs, secretPath.path, {
      ...(Array.isArray(currentSecret) ? {
        ids: currentSecret.map(({
          id
        }) => id)
      } : {
        id: currentSecret.id
      }),
      ...(typeof maybeHash === 'string' && {
        hash: maybeHash
      })
    });
  });
  return {
    soObjectWithSecrets: objectWithSecretRefs,
    secretReferences: secrets.reduce((acc, secret) => {
      if (Array.isArray(secret)) {
        return [...acc, ...secret.map(({
          id
        }) => ({
          id
        }))];
      }
      return [...acc, {
        id: secret.id
      }];
    }, [])
  };
}

/**
 * Takes a generic object T to update and its old and new secret paths
 * Updates secrets and returns the references
 */
async function extractAndUpdateSOSecrets(opts) {
  const {
    updatedSoObject,
    oldSecretPaths,
    updatedSecretPaths,
    esClient,
    secretHashes
  } = opts;
  if (!oldSecretPaths.length && !updatedSecretPaths.length) {
    return {
      updatedSoObject,
      secretReferences: [],
      secretsToDelete: []
    };
  }
  const {
    toCreate,
    toDelete,
    noChange
  } = diffSOSecretPaths(oldSecretPaths, updatedSecretPaths);
  const createdSecrets = await createSecrets({
    esClient,
    values: toCreate.map(secretPath => secretPath.value)
  });
  const soObjectWithSecretRefs = JSON.parse(JSON.stringify(updatedSoObject));
  toCreate.forEach((secretPath, i) => {
    const pathWithoutPrefix = secretPath.path.replace('secrets.', '');
    const maybeHash = (0, _lodash.get)(secretHashes, pathWithoutPrefix);
    const currentSecret = createdSecrets[i];
    (0, _saferLodashSet.set)(soObjectWithSecretRefs, secretPath.path, {
      ...(Array.isArray(currentSecret) ? {
        ids: currentSecret.map(({
          id
        }) => id)
      } : {
        id: currentSecret.id
      }),
      ...(typeof maybeHash === 'string' && {
        hash: maybeHash
      })
    });
  });
  const secretReferences = [...noChange.reduce((acc, secretPath) => {
    const currentValue = secretPath.value;
    if ('ids' in currentValue) {
      return [...acc, ...currentValue.ids.map(id => ({
        id
      }))];
    } else {
      return [...acc, {
        id: currentValue.id
      }];
    }
  }, []), ...createdSecrets.reduce((acc, secret) => {
    if (Array.isArray(secret)) {
      return [...acc, ...secret.map(({
        id
      }) => ({
        id
      }))];
    }
    return [...acc, {
      id: secret.id
    }];
  }, [])];
  return {
    updatedSoObject: soObjectWithSecretRefs,
    secretReferences,
    secretsToDelete: toDelete.map(secretPath => ({
      id: secretPath.value.id
    }))
  };
}

/**
 * Makes the diff betwwen old and new secrets paths
 */
function diffSOSecretPaths(oldPaths, newPaths) {
  const toCreate = [];
  const toDelete = [];
  const noChange = [];
  const newPathsByPath = (0, _lodash.keyBy)(newPaths, 'path');
  for (const oldPath of oldPaths) {
    if (!newPathsByPath[oldPath.path]) {
      toDelete.push(oldPath);
    }
    const newPath = newPathsByPath[oldPath.path];
    if (newPath && newPath.value) {
      const newValue = newPath.value;
      if (typeof newValue === 'string') {
        toCreate.push(newPath);
        toDelete.push(oldPath);
      } else {
        noChange.push(newPath);
      }
    }
    delete newPathsByPath[oldPath.path];
  }
  const remainingNewPaths = Object.values(newPathsByPath);
  return {
    toCreate: [...toCreate, ...remainingNewPaths],
    toDelete,
    noChange
  };
}