"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseFileStorageIndex = void 0;
var _services = require("../../../common/services");
var _constants = require("../../../common/constants");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Given a document index (from either a file's metadata doc or a file's chunk doc), utility will
 * parse it and return information about that index
 * @param index
 */
const parseFileStorageIndex = index => {
  const response = {
    index: '',
    integration: '',
    type: 'meta',
    direction: 'from-host'
  };
  const fileStorageIndexPatterns = [_constants.FILE_STORAGE_METADATA_INDEX_PATTERN, _constants.FILE_STORAGE_DATA_INDEX_PATTERN, _constants.FILE_STORAGE_TO_HOST_METADATA_INDEX_PATTERN, _constants.FILE_STORAGE_TO_HOST_DATA_INDEX_PATTERN];
  for (const indexPattern of fileStorageIndexPatterns) {
    const indexPrefix = indexPattern.substring(0, indexPattern.indexOf('*'));
    if (index.includes(indexPrefix)) {
      const isDeliveryToHost = index.includes('-tohost-');
      const isDataIndex = index.includes('host-data-');
      const integrationPosition = indexPattern.split('-').indexOf('*');
      const integration = index.replace(/^\.ds-/, '').split('-').at(integrationPosition);
      if (!integration) {
        throw new _errors.FleetError(`Index name ${index} does not seem to be a File storage index`);
      }
      response.direction = isDeliveryToHost ? 'to-host' : 'from-host';
      response.type = isDataIndex ? 'data' : 'meta';
      response.integration = integration;
      response.index = isDataIndex ? (0, _services.getFileDataIndexName)(response.integration, isDeliveryToHost) : (0, _services.getFileMetadataIndexName)(response.integration, isDeliveryToHost);
      return response;
    }
  }
  throw new _errors.FleetError(`Unable to parse index [${index}]. Does not match a known index pattern: [${fileStorageIndexPatterns.join(' | ')}]`);
};
exports.parseFileStorageIndex = parseFileStorageIndex;