"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withPackageSpan = exports.setLastUploadInstallCache = exports.resolveDataStreamsMap = exports.resolveDataStreamFields = exports.getLastUploadInstallCache = void 0;
var _apmUtils = require("@kbn/apm-utils");
var _jsYaml = require("js-yaml");
var _app_context = require("../../app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const withPackageSpan = (stepName, func) => (0, _apmUtils.withSpan)({
  name: stepName,
  type: 'package'
}, func);
exports.withPackageSpan = withPackageSpan;
const normalizeFields = (fields, prefix = '') => {
  return fields.reduce((normalizedFields, field) => {
    const flatName = prefix ? `${prefix}.${field.name}` : field.name;
    // Recursively resolve field groups
    if (isGroupField(field)) {
      return Object.assign(normalizedFields, normalizeFields(field.fields || [], flatName));
    }
    normalizedFields[flatName] = createIntegrationField(field, flatName);
    return normalizedFields;
  }, {});
};
const createIntegrationField = (field, flatName) => ({
  ...field,
  flat_name: flatName
});
const isGroupField = field => {
  return field.type === 'group';
};
const resolveDataStreamsMap = dataStreams => {
  if (!dataStreams) return new Map();
  return dataStreams.reduce((dataStreamsMap, dataStream) => {
    dataStreamsMap.set(dataStream.dataset, dataStream);
    return dataStreamsMap;
  }, new Map());
};
exports.resolveDataStreamsMap = resolveDataStreamsMap;
const resolveDataStreamFields = ({
  dataStream,
  assetsMap,
  excludedFieldsAssets
}) => {
  const {
    dataset,
    path
  } = dataStream;
  const dataStreamFieldsAssetPaths = getDataStreamFieldsAssetPaths(assetsMap, path, excludedFieldsAssets);

  /**
   * We want to create a single dictionary with fields taken from all the dataset /fields assets.
   * This step
   * - reads the files buffer
   * - normalizes the fields data structure for each file
   * - finally merge the fields from each file into a single dictionary
   */
  const fields = dataStreamFieldsAssetPaths.reduce((dataStreamFields, fieldsAssetPath) => {
    const fieldsAssetBuffer = assetsMap.get(fieldsAssetPath);
    if (fieldsAssetBuffer) {
      const fieldsAssetJSON = (0, _jsYaml.load)(fieldsAssetBuffer.toString('utf8'));
      const normalizedFields = normalizeFields(fieldsAssetJSON);
      Object.assign(dataStreamFields, normalizedFields);
    }
    return dataStreamFields;
  }, {});
  return {
    [dataset]: fields
  };
};
exports.resolveDataStreamFields = resolveDataStreamFields;
const isFieldsAsset = (assetPath, dataStreamPath, excludedFieldsAssets = []) => {
  return new RegExp(`.*\/data_stream\/${dataStreamPath}\/fields\/(?!(${excludedFieldsAssets.join('|')})$).*\.yml`, 'i').test(assetPath);
};
const getDataStreamFieldsAssetPaths = (assetsMap, dataStreamPath, excludedFieldsAssets) => {
  return [...assetsMap.keys()].filter(path => isFieldsAsset(path, dataStreamPath, excludedFieldsAssets));
};

// Set an in memory cache to save the timestamp of latest install by upload
const lastInstalledByUpload = new Map();
const setLastUploadInstallCache = () => {
  const logger = _app_context.appContextService.getLogger();
  const key = 'upload';
  const time = Date.now();
  logger.debug(`Setting timestamp ${time} to cache for install by ${key}`);
  return lastInstalledByUpload.set(key, time);
};
exports.setLastUploadInstallCache = setLastUploadInstallCache;
const getLastUploadInstallCache = () => {
  return lastInstalledByUpload.get('upload');
};
exports.getLastUploadInstallCache = getLastUploadInstallCache;