"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._purgeBundledPackagesCache = _purgeBundledPackagesCache;
exports.getBundledPackageByName = getBundledPackageByName;
exports.getBundledPackageByPkgKey = getBundledPackageByPkgKey;
exports.getBundledPackageForInstallation = getBundledPackageForInstallation;
exports.getBundledPackages = getBundledPackages;
var _promises = _interopRequireDefault(require("fs/promises"));
var _path = _interopRequireDefault(require("path"));
var _lodash = require("lodash");
var _errors = require("../../../errors");
var _app_context = require("../../app_context");
var _registry = require("../registry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

let CACHE_BUNDLED_PACKAGES;
function _purgeBundledPackagesCache() {
  CACHE_BUNDLED_PACKAGES = undefined;
}
function bundledPackagesFromCache() {
  if (!CACHE_BUNDLED_PACKAGES) {
    throw new Error('CACHE_BUNDLED_PACKAGES is not populated');
  }
  return CACHE_BUNDLED_PACKAGES.map(({
    name,
    version,
    getBuffer
  }) => ({
    name,
    version,
    getBuffer: (0, _lodash.once)(getBuffer)
  }));
}
async function getBundledPackages() {
  var _config$developer, _config$developer2;
  const config = _app_context.appContextService.getConfig();
  if ((config === null || config === void 0 ? void 0 : (_config$developer = config.developer) === null || _config$developer === void 0 ? void 0 : _config$developer.disableBundledPackagesCache) !== true && CACHE_BUNDLED_PACKAGES) {
    return bundledPackagesFromCache();
  }
  const bundledPackageLocation = config === null || config === void 0 ? void 0 : (_config$developer2 = config.developer) === null || _config$developer2 === void 0 ? void 0 : _config$developer2.bundledPackageLocation;
  if (!bundledPackageLocation) {
    throw new _errors.BundledPackageLocationNotFoundError('xpack.fleet.developer.bundledPackageLocation is not configured');
  }

  // If the bundled package directory is missing, we log a warning during setup,
  // so we can safely ignore this case here and just retun and empty array
  try {
    await _promises.default.stat(bundledPackageLocation);
  } catch (error) {
    return [];
  }
  try {
    const dirContents = await _promises.default.readdir(bundledPackageLocation);
    const zipFiles = dirContents.filter(file => file.endsWith('.zip'));
    const result = await Promise.all(zipFiles.map(async zipFile => {
      const {
        pkgName,
        pkgVersion
      } = (0, _registry.splitPkgKey)(zipFile.replace(/\.zip$/, ''));
      const getBuffer = () => _promises.default.readFile(_path.default.join(bundledPackageLocation, zipFile));
      return {
        name: pkgName,
        version: pkgVersion,
        getBuffer
      };
    }));
    CACHE_BUNDLED_PACKAGES = result;
    return bundledPackagesFromCache();
  } catch (err) {
    const logger = _app_context.appContextService.getLogger();
    logger.warn(`Unable to read bundled packages from ${bundledPackageLocation}`);
    return [];
  }
}
async function getBundledPackageForInstallation(installation) {
  const bundledPackages = await getBundledPackages();
  return bundledPackages.find(bundledPkg => bundledPkg.name === installation.name && bundledPkg.version === installation.version);
}
async function getBundledPackageByPkgKey(pkgKey) {
  const bundledPackages = await getBundledPackages();
  return bundledPackages.find(pkg => {
    if (pkgKey.includes('-')) {
      return (0, _registry.pkgToPkgKey)(pkg) === pkgKey;
    } else {
      return pkg.name === pkgKey;
    }
  });
}
async function getBundledPackageByName(name) {
  const bundledPackages = await getBundledPackages();
  return bundledPackages.find(pkg => pkg.name === name);
}