"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteAgentUploadFile = deleteAgentUploadFile;
exports.getAgentUploadFile = getAgentUploadFile;
exports.getAgentUploads = getAgentUploads;
exports.getDownloadHeadersForFile = getDownloadHeadersForFile;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/files-plugin/server");
var _app_context = require("../app_context");
var _common = require("../../../common");
var _constants = require("../../constants");
var _files = require("../files");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getAgentUploads(esClient, agentId) {
  const getFile = async actionId => {
    try {
      var _fileResponse$hits$hi;
      const fileResponse = await esClient.search({
        index: _constants.FILE_STORAGE_METADATA_AGENT_INDEX,
        query: {
          bool: {
            filter: {
              bool: {
                must: [{
                  term: {
                    agent_id: agentId
                  }
                }, {
                  term: {
                    action_id: actionId
                  }
                }]
              }
            }
          }
        }
      });
      if (fileResponse.hits.hits.length === 0) {
        _app_context.appContextService.getLogger().trace(`No matches for action_id ${actionId} and agent_id ${agentId}`);
        return;
      }
      return {
        id: fileResponse.hits.hits[0]._id,
        ...((_fileResponse$hits$hi = fileResponse.hits.hits[0]._source) === null || _fileResponse$hits$hi === void 0 ? void 0 : _fileResponse$hits$hi.file)
      };
    } catch (err) {
      if (err.statusCode === 404) {
        _app_context.appContextService.getLogger().debug(err);
        return;
      } else {
        throw err;
      }
    }
  };
  const actions = await _getRequestDiagnosticsActions(esClient, agentId);
  const results = [];
  for (const action of actions) {
    var _file$name, _file$Status, _file$id;
    const file = await getFile(action.actionId);

    // File list is initially built from list of diagnostic actions.
    // If file was deleted intentially by ILM policy or user based on the meta information,
    // or if the meta information does not exist AND the action is old (new actions are
    // ok to show because we want to show in progress files)
    // skip returning this action/file information.
    if ((file === null || file === void 0 ? void 0 : file.Status) === 'DELETED' || !file && Date.parse(action.timestamp) < Date.now() - 89 * 24 * 3600 * 1000) {
      continue;
    }
    const fileName = (_file$name = file === null || file === void 0 ? void 0 : file.name) !== null && _file$name !== void 0 ? _file$name : `elastic-agent-diagnostics-${_moment.default.utc(action.timestamp).format('YYYY-MM-DDTHH-mm-ss')}Z-00.zip`;
    const filePath = file ? _common.agentRouteService.getAgentFileDownloadLink(file.id, file.name) : '';
    const isActionExpired = action.expiration ? Date.parse(action.expiration) < Date.now() : false;
    const status = (_file$Status = file === null || file === void 0 ? void 0 : file.Status) !== null && _file$Status !== void 0 ? _file$Status : isActionExpired ? 'EXPIRED' : action.error ? 'FAILED' : 'IN_PROGRESS';
    const result = {
      actionId: action.actionId,
      id: (_file$id = file === null || file === void 0 ? void 0 : file.id) !== null && _file$id !== void 0 ? _file$id : action.actionId,
      status,
      name: fileName,
      createTime: action.timestamp,
      filePath,
      error: action.error
    };
    results.push(result);
  }
  return results;
}
async function _getRequestDiagnosticsActions(esClient, agentId) {
  const agentActionRes = await esClient.search({
    index: _common.AGENT_ACTIONS_INDEX,
    ignore_unavailable: true,
    size: _constants.SO_SEARCH_LIMIT,
    sort: {
      '@timestamp': 'desc'
    },
    query: {
      bool: {
        must: [{
          term: {
            type: 'REQUEST_DIAGNOSTICS'
          }
        }, {
          term: {
            agents: agentId
          }
        }]
      }
    }
  });
  const agentActions = agentActionRes.hits.hits.map(hit => {
    var _hit$_source, _hit$_source2, _hit$_source3;
    return {
      actionId: (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.action_id,
      timestamp: (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2['@timestamp'],
      expiration: (_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : _hit$_source3.expiration
    };
  });
  if (agentActions.length === 0) {
    return [];
  }
  try {
    const actionResultsRes = await esClient.search({
      index: _common.AGENT_ACTIONS_RESULTS_INDEX,
      ignore_unavailable: true,
      size: _constants.SO_SEARCH_LIMIT,
      query: {
        bool: {
          must: [{
            terms: {
              action_id: agentActions.map(action => action.actionId)
            }
          }, {
            term: {
              agent_id: agentId
            }
          }]
        }
      }
    });
    const actionResults = actionResultsRes.hits.hits.map(hit => {
      var _hit$_source4, _hit$_source5, _hit$_source6, _hit$_source6$data, _hit$_source7;
      return {
        actionId: (_hit$_source4 = hit._source) === null || _hit$_source4 === void 0 ? void 0 : _hit$_source4.action_id,
        timestamp: (_hit$_source5 = hit._source) === null || _hit$_source5 === void 0 ? void 0 : _hit$_source5['@timestamp'],
        fileId: (_hit$_source6 = hit._source) === null || _hit$_source6 === void 0 ? void 0 : (_hit$_source6$data = _hit$_source6.data) === null || _hit$_source6$data === void 0 ? void 0 : _hit$_source6$data.upload_id,
        error: (_hit$_source7 = hit._source) === null || _hit$_source7 === void 0 ? void 0 : _hit$_source7.error
      };
    });
    return agentActions.map(action => {
      const actionResult = actionResults.find(result => result.actionId === action.actionId);
      return {
        actionId: action.actionId,
        timestamp: action.timestamp,
        expiration: action.expiration,
        fileId: actionResult === null || actionResult === void 0 ? void 0 : actionResult.fileId,
        error: actionResult === null || actionResult === void 0 ? void 0 : actionResult.error
      };
    });
  } catch (err) {
    if (err.statusCode === 404) {
      // .fleet-actions-results does not yet exist
      _app_context.appContextService.getLogger().debug(err);
      return [];
    } else {
      throw err;
    }
  }
}
async function getAgentUploadFile(esClient, id, fileName) {
  try {
    const fileClient = (0, _server.createEsFileClient)({
      blobStorageIndex: _constants.FILE_STORAGE_DATA_AGENT_INDEX,
      metadataIndex: _constants.FILE_STORAGE_METADATA_AGENT_INDEX,
      elasticsearchClient: esClient,
      logger: _app_context.appContextService.getLogger(),
      indexIsAlias: true
    });
    const file = await fileClient.get({
      id
    });
    return {
      body: await file.downloadContent(),
      headers: getDownloadHeadersForFile(fileName)
    };
  } catch (error) {
    _app_context.appContextService.getLogger().error(error);
    throw error;
  }
}
async function deleteAgentUploadFile(esClient, id) {
  try {
    // We manually delete the documents from the data streams with `_delete_by_query`
    // because data streams do not support single deletes. See:
    // https://www.elastic.co/guide/en/elasticsearch/reference/current/data-streams.html#data-streams-append-only

    // Delete the file from the file storage data stream
    const filesDeleteResponse = await esClient.deleteByQuery({
      index: _constants.FILE_STORAGE_DATA_AGENT_INDEX,
      refresh: true,
      query: {
        match: {
          bid: id // Use `bid` instead of `_id` because `_id` has additional suffixes
        }
      }
    });
    if (!!(filesDeleteResponse.deleted && filesDeleteResponse.deleted > 0 || filesDeleteResponse.total === 0)) {
      // Update the metadata to mark the file as deleted
      const updateMetadataStatusResponse = (await (0, _files.updateFilesStatus)(esClient, undefined, {
        [_constants.FILE_STORAGE_METADATA_AGENT_INDEX]: new Set([id])
      }, 'DELETED'))[0];
      if (updateMetadataStatusResponse.total === 0) {
        throw new _errors.FleetError(`Failed to update file ${id} metadata`);
      }
      return {
        id,
        deleted: true
      };
    } else {
      throw new _errors.FleetError(`Failed to delete file ${id} from file storage data stream`);
    }
  } catch (error) {
    _app_context.appContextService.getLogger().error(error);
    throw error;
  }
}
function getDownloadHeadersForFile(fileName) {
  return {
    'content-type': 'application/octet-stream',
    // Note, this name can be overridden by the client if set via a "download" attribute on the HTML tag.
    'content-disposition': `attachment; filename="${fileName}"`,
    'cache-control': 'max-age=31536000, immutable',
    // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options
    'x-content-type-options': 'nosniff'
  };
}