"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateOtelcolConfig = generateOtelcolConfig;
var _constants = require("../../../common/constants");
var _errors = require("../../errors");
var _output_helpers = require("../../../common/services/output_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Generate OTel Collector policy
function generateOtelcolConfig(inputs, dataOutput) {
  const otelConfigs = inputs.filter(input => input.type === _constants.OTEL_COLLECTOR_INPUT_TYPE).flatMap(input => {
    var _input$streams;
    const otelInputs = ((_input$streams = input === null || input === void 0 ? void 0 : input.streams) !== null && _input$streams !== void 0 ? _input$streams : []).map(stream => {
      // Avoid dots in keys, as they can create subobjects in agent config.
      const suffix = (input.id + '-' + stream.id).replaceAll('.', '-');
      const attributesTransform = generateOTelAttributesTransform(stream.data_stream.type ? stream.data_stream.type : 'logs', stream.data_stream.dataset, 'data_stream' in input ? input.data_stream.namespace : 'default', suffix);
      return appendOtelComponents({
        ...addSuffixToOtelcolComponentsConfig('extensions', suffix, stream === null || stream === void 0 ? void 0 : stream.extensions),
        ...addSuffixToOtelcolComponentsConfig('receivers', suffix, stream === null || stream === void 0 ? void 0 : stream.receivers),
        ...addSuffixToOtelcolComponentsConfig('processors', suffix, stream === null || stream === void 0 ? void 0 : stream.processors),
        ...addSuffixToOtelcolComponentsConfig('connectors', suffix, stream === null || stream === void 0 ? void 0 : stream.connectors),
        ...addSuffixToOtelcolComponentsConfig('exporters', suffix, stream === null || stream === void 0 ? void 0 : stream.exporters),
        ...(stream !== null && stream !== void 0 && stream.service ? {
          service: {
            ...stream.service,
            ...addSuffixToOtelcolComponentsConfig('pipelines', suffix, addSuffixToOtelcolPipelinesComponents(stream.service.pipelines, suffix))
          }
        } : {})
      }, 'processors', [attributesTransform]);
    });
    return otelInputs;
  });
  if (otelConfigs.length === 0) {
    return {};
  }
  const config = mergeOtelcolConfigs(otelConfigs);
  return attachOtelcolExporter(config, dataOutput);
}
function generateOTelAttributesTransform(type, dataset, namespace, suffix) {
  let otelType;
  let context;
  switch (type) {
    case 'logs':
      otelType = 'log';
      context = 'log';
      break;
    case 'metrics':
      otelType = 'metric';
      context = 'datapoint';
      break;
    case 'traces':
      otelType = 'trace';
      context = 'span';
      break;
    default:
      throw new _errors.FleetError(`unexpected data stream type ${type}`);
  }
  return {
    [`transform/${suffix}-routing`]: {
      [`${otelType}_statements`]: [{
        context,
        statements: [`set(attributes["data_stream.type"], "${type}")`, `set(attributes["data_stream.dataset"], "${dataset}")`, `set(attributes["data_stream.namespace"], "${namespace}")`]
      }]
    }
  };
}
function appendOtelComponents(config, type, components) {
  components.forEach(component => {
    var _Object$entries$find, _config$service;
    Object.assign(config, config, {
      [type]: {
        ...((_Object$entries$find = Object.entries(config).find(([key]) => key === type)) === null || _Object$entries$find === void 0 ? void 0 : _Object$entries$find[1]),
        ...component
      }
    });
    if ((_config$service = config.service) !== null && _config$service !== void 0 && _config$service.pipelines) {
      Object.values(config.service.pipelines).forEach(pipeline => {
        Object.keys(component).forEach(id => {
          pipeline.processors = (pipeline.processors ? pipeline.processors : []).concat([id]);
        });
      });
    }
  });
  return config;
}
function addSuffixToOtelcolComponentsConfig(type, suffix, components) {
  if (!components) {
    return {};
  }
  const generated = {};
  Object.entries(components).forEach(([id, config]) => {
    generated[id + '/' + suffix] = {
      ...config
    };
  });
  return {
    [type]: generated
  };
}
function addSuffixToOtelcolPipelinesComponents(pipelines, suffix) {
  const result = {};
  Object.entries(pipelines).forEach(([pipelineID, pipeline]) => {
    const newPipeline = {};
    Object.entries(pipeline).forEach(([type, componentIDs]) => {
      newPipeline[type] = componentIDs.map(id => id + '/' + suffix);
    });
    result[pipelineID] = newPipeline;
  });
  return result;
}
function mergeOtelcolConfigs(otelConfigs) {
  return otelConfigs.reduce((merged, next) => {
    var _next$service, _merged$service, _merged$service2, _next$service2;
    if (!next) {
      return merged;
    }
    const extensions = {
      ...merged.extensions,
      ...next.extensions
    };
    const receivers = {
      ...merged.receivers,
      ...next.receivers
    };
    const processors = {
      ...merged.processors,
      ...next.processors
    };
    const connectors = {
      ...merged.connectors,
      ...next.connectors
    };
    const exporters = {
      ...merged.exporters,
      ...next.exporters
    };
    return {
      ...(Object.keys(extensions).length > 0 ? {
        extensions
      } : {}),
      ...(Object.keys(receivers).length > 0 ? {
        receivers
      } : {}),
      ...(Object.keys(processors).length > 0 ? {
        processors
      } : {}),
      ...(Object.keys(connectors).length > 0 ? {
        connectors
      } : {}),
      ...(Object.keys(exporters).length > 0 ? {
        exporters
      } : {}),
      service: {
        ...merged.service,
        ...((_next$service = next.service) !== null && _next$service !== void 0 && _next$service.extensions ? {
          extensions: ((_merged$service = merged.service) !== null && _merged$service !== void 0 && _merged$service.extensions ? merged.service.extensions : []).concat(next.service.extensions)
        } : {}),
        pipelines: {
          ...((_merged$service2 = merged.service) === null || _merged$service2 === void 0 ? void 0 : _merged$service2.pipelines),
          ...((_next$service2 = next.service) === null || _next$service2 === void 0 ? void 0 : _next$service2.pipelines)
        }
      }
    };
  });
}
function attachOtelcolExporter(config, dataOutput) {
  var _config$service2;
  if (!dataOutput) {
    return config;
  }
  const exporter = generateOtelcolExporter(dataOutput);
  config.connectors = {
    ...config.connectors,
    forward: {}
  };
  config.exporters = {
    ...config.exporters,
    ...exporter
  };
  if ((_config$service2 = config.service) !== null && _config$service2 !== void 0 && _config$service2.pipelines) {
    const signalTypes = new Set();
    Object.entries(config.service.pipelines).forEach(([id, pipeline]) => {
      config.service.pipelines[id] = {
        ...pipeline,
        exporters: [...(pipeline.exporters || []), 'forward']
      };
      signalTypes.add(signalType(id));
    });
    signalTypes.forEach(id => {
      config.service.pipelines[id] = {
        receivers: ['forward'],
        exporters: Object.keys(exporter)
      };
    });
  }
  return config;
}
function generateOtelcolExporter(dataOutput) {
  switch (dataOutput.type) {
    case _constants.outputType.Elasticsearch:
      const outputID = (0, _output_helpers.getOutputIdForAgentPolicy)(dataOutput);
      return {
        [`elasticsearch/${outputID}`]: {
          endpoints: dataOutput.hosts
        }
      };
    default:
      throw new _errors.FleetError(`output type ${dataOutput.type} not supported when policy contains OTel inputs`);
  }
}
function signalType(id) {
  return id.substring(0, id.indexOf('/'));
}