"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAPIRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../common/constants");
var _security = require("../../services/security");
var _experimental_features = require("../../../common/experimental_features");
var _constants2 = require("../../constants");
var _types = require("../../types");
var AgentService = _interopRequireWildcard(require("../../services/agents"));
var _agent = require("../../types/rest_spec/agent");
var _api_privileges = require("../../constants/api_privileges");
var _security2 = require("../../services/security/security");
var _errors = require("../schema/errors");
var _handlers = require("./handlers");
var _actions_handlers = require("./actions_handlers");
var _unenroll_handler = require("./unenroll_handler");
var _upgrade_handler = require("./upgrade_handler");
var _request_diagnostics_handler = require("./request_diagnostics_handler");
var _migrate_handlers = require("./migrate_handlers");
var _change_privilege_level_handlers = require("./change_privilege_level_handlers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerAPIRoutes = (router, config) => {
  const experimentalFeatures = (0, _experimental_features.parseExperimentalConfigValue)(config.enableExperimental);
  // Get one
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get an agent`,
    description: `Get an agent by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneAgentRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.GetAgentResponseSchema
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _handlers.getAgentHandler);

  // Migrate
  // Single agent migration
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.MIGRATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Migrate a single agent`,
    description: `Migrate a single agent to another cluster.`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.MigrateSingleAgentRequestSchema,
      response: {
        200: {
          body: () => _types.MigrateSingleAgentResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _migrate_handlers.migrateSingleAgentHandler);

  // Bulk migrate multiple agents
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.BULK_MIGRATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Migrate multiple agents`,
    description: `Bulk migrate agents to another cluster.`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.BulkMigrateAgentsRequestSchema,
      response: {
        200: {
          body: () => _types.BulkMigrateAgentsResponseSchema,
          description: 'OK: A successful request.'
        },
        400: {
          body: _errors.genericErrorResponse,
          description: 'A bad request.'
        }
      }
    }
  }, _migrate_handlers.bulkMigrateAgentsHandler);

  // Update
  router.versioned.put({
    path: _constants2.AGENT_API_ROUTES.UPDATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Update an agent by ID`,
    description: `Update an agent by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.UpdateAgentRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.GetAgentResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.updateAgentHandler);

  // Bulk Update Tags
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.BULK_UPDATE_AGENT_TAGS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Bulk update agent tags`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _agent.PostBulkUpdateAgentTagsRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostBulkActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.bulkUpdateAgentTagsHandler);

  // Delete
  router.versioned.delete({
    path: _constants2.AGENT_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Delete an agent`,
    description: `Delete an agent by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeleteAgentRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.DeleteAgentResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.deleteAgentHandler);

  // List
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get agents`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetAgentsRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.GetAgentsResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentsHandler);

  // List Agent Tags
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.LIST_TAGS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get agent tags`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetTagsRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetTagsResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentTagsHandler);

  // Agent actions
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.ACTIONS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Create an agent action`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostNewAgentActionRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostNewAgentActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, (0, _actions_handlers.postNewAgentActionHandlerBuilder)({
    getAgent: AgentService.getAgentById,
    cancelAgentAction: AgentService.cancelAgentAction,
    createAgentAction: AgentService.createAgentAction,
    getAgentActions: AgentService.getAgentActions
  }));
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.CANCEL_ACTIONS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Cancel an agent action`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _agent.PostCancelActionRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostNewAgentActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, (0, _actions_handlers.postCancelActionHandlerBuilder)({
    getAgent: AgentService.getAgentById,
    cancelAgentAction: AgentService.cancelAgentAction,
    createAgentAction: AgentService.createAgentAction,
    getAgentActions: AgentService.getAgentActions
  }));

  // Get agents by Action_Ids
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get agents by action ids`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostRetrieveAgentsByActionsRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostRetrieveAgentsByActionsResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.postRetrieveAgentsByActionsHandler);
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.UNENROLL_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Unenroll an agent`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostAgentUnenrollRequestSchema,
      response: {}
    }
  }, _unenroll_handler.postAgentUnenrollHandler);
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.REASSIGN_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Reassign an agent`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostAgentReassignRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.object({})
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.postAgentReassignHandler);
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.REQUEST_DIAGNOSTICS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Request agent diagnostics`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostRequestDiagnosticsActionRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostBulkActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _request_diagnostics_handler.requestDiagnosticsHandler);
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.BULK_REQUEST_DIAGNOSTICS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Bulk request diagnostics from agents`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostBulkRequestDiagnosticsActionRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostBulkActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _request_diagnostics_handler.bulkRequestDiagnosticsHandler);
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.LIST_UPLOADS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get agent uploads`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.ListAgentUploadsRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.ListAgentUploadsResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentUploadsHandler);
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.GET_UPLOAD_FILE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get an uploaded file`,
    description: `Get a file uploaded by an agent.`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetAgentUploadFileRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.stream() // Readable
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentUploadFileHandler);
  router.versioned.delete({
    path: _constants2.AGENT_API_ROUTES.DELETE_UPLOAD_FILE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Delete an uploaded file`,
    description: `Delete a file uploaded by an agent.`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeleteAgentUploadFileRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.DeleteAgentUploadFileResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.deleteAgentUploadFileHandler);
  // Get agent status for policy
  router.versioned
  // @ts-ignore  https://github.com/elastic/kibana/issues/203170
  .get({
    path: _constants2.AGENT_API_ROUTES.STATUS_PATTERN,
    // TODO move to kibana authz https://github.com/elastic/kibana/issues/203170
    fleetAuthz: fleetAuthz => (0, _security2.calculateRouteAuthz)(fleetAuthz, (0, _security.getRouteRequiredAuthz)('get', _constants2.AGENT_API_ROUTES.STATUS_PATTERN)).granted,
    summary: `Get an agent status summary`,
    options: {
      tags: ['oas-tag:Elastic Agent status']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetAgentStatusRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.GetAgentStatusResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentStatusForAgentPolicyHandler);
  // Agent data
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.DATA_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get incoming agent data`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetAgentDataRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.GetAgentDataResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentDataHandler);

  // upgrade agent
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.UPGRADE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Upgrade an agent`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostAgentUpgradeRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.object({})
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _upgrade_handler.postAgentUpgradeHandler);
  // bulk upgrade
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.BULK_UPGRADE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Bulk upgrade agents`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostBulkAgentUpgradeRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostBulkActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _upgrade_handler.postBulkAgentsUpgradeHandler);

  // Current actions
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.ACTION_STATUS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get an agent action status`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetActionStatusRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.GetActionStatusResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getActionStatusHandler);

  // Bulk reassign
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.BULK_REASSIGN_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Bulk reassign agents`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostBulkAgentReassignRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostBulkActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.postBulkAgentReassignHandler);

  // Bulk unenroll
  router.versioned.post({
    path: _constants2.AGENT_API_ROUTES.BULK_UNENROLL_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Bulk unenroll agents`,
    options: {
      tags: ['oas-tag:Elastic Agent actions']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostBulkAgentUnenrollRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.PostBulkActionResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _unenroll_handler.postBulkAgentsUnenrollHandler);

  // Available versions for upgrades
  router.versioned.get({
    path: _constants2.AGENT_API_ROUTES.AVAILABLE_VERSIONS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Get available agent versions`,
    options: {
      tags: ['oas-tag:Elastic Agents']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _agent.GetAvailableAgentVersionsResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAvailableVersionsHandler);

  // route used by export CSV feature on the UI to generate report
  router.versioned.get({
    path: '/internal/fleet/agents/status_runtime_field',
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.internal.v1,
    validate: {
      request: {},
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.string()
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentStatusRuntimeFieldHandler);

  // Change agent privilege level
  if (experimentalFeatures.enableAgentPrivilegeLevelChange) {
    router.versioned.post({
      path: _constants2.AGENT_API_ROUTES.PRIVILEGE_LEVEL_CHANGE_PATTERN,
      security: {
        authz: {
          requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
        }
      },
      summary: `Change agent privilege level`,
      description: `Change the privilege level of a single agent to unprivileged.`,
      options: {
        tags: ['oas-tag:Elastic Agents'],
        availability: {
          since: '9.2.0',
          stability: 'experimental'
        }
      }
    }).addVersion({
      version: _constants.API_VERSIONS.public.v1,
      validate: {
        request: _agent.ChangeAgentPrivilegeLevelRequestSchema,
        response: {
          200: {
            description: 'OK: A successful request.',
            body: () => _agent.ChangeAgentPrivilegeLevelResponseSchema
          },
          400: {
            description: 'A bad request.',
            body: _errors.genericErrorResponse
          }
        }
      }
    }, _change_privilege_level_handlers.changeAgentPrivilegeLevelHandler);
  }
};
exports.registerAPIRoutes = registerAPIRoutes;