"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OtelFieldsRepository = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _PathReporter = require("io-ts/PathReporter");
var _lodash = require("lodash");
var _Either = require("fp-ts/Either");
var _fields_metadata = require("../../../../common/fields_metadata");
var _fields_metadata_dictionary = require("../../../../common/fields_metadata/models/fields_metadata_dictionary");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Helper function to convert structured semconv fields to FieldMetadataPlain format
function convertSemconvToFieldMetadata(semconvFields) {
  const result = {};
  for (const [fieldName, fieldData] of Object.entries(semconvFields)) {
    // Ensure fieldData is an object before proceeding
    if (typeof fieldData !== 'object' || fieldData === null) {
      throw new Error(`Invalid field data for ${fieldName}: expected object, got ${typeof fieldData}`);
    }
    result[fieldName] = {
      name: fieldData.name,
      description: fieldData.description,
      type: fieldData.type,
      ...('example' in fieldData && {
        example: fieldData.example
      })
    };
  }
  return result;
}
const OTEL_PREFIXES_TO_STRIP = ['resource.attributes.', 'scope.attributes.', 'attributes.'];

/**
 * Strip common OpenTelemetry prefixes from field names to match the actual field names
 * in the semantic conventions dictionary.
 *
 * Examples:
 * - "resource.attributes.cloud.account.id" -> "cloud.account.id"
 * - "scope.attributes.service.name" -> "service.name"
 * - "attributes.service.name" -> "service.name"
 * - "cloud.account.id" -> "cloud.account.id" (no change)
 */
function stripOtelPrefixes(fieldName) {
  for (const prefix of OTEL_PREFIXES_TO_STRIP) {
    if (fieldName.startsWith(prefix)) {
      return fieldName.substring(prefix.length);
    }
  }

  // Return original field name if no prefixes match
  return fieldName;
}
class OtelFieldsRepository {
  constructor(otelFields) {
    (0, _defineProperty2.default)(this, "otelFields", void 0);
    this.otelFields = (0, _lodash.mapValues)(otelFields, field => _common.FieldMetadata.create({
      ...field,
      source: 'otel'
    }));
  }
  getByName(fieldName) {
    // Strip OTel prefixes before looking up the field
    const strippedFieldName = stripOtelPrefixes(fieldName);
    return this.otelFields[strippedFieldName];
  }
  find({
    fieldNames
  } = {}) {
    if (!fieldNames) {
      return _fields_metadata_dictionary.FieldsMetadataDictionary.create(this.otelFields);
    }
    const fields = fieldNames.reduce((fieldsMetadata, fieldName) => {
      const field = this.getByName(fieldName);
      if (field) {
        fieldsMetadata[fieldName] = field;
      }
      return fieldsMetadata;
    }, {});
    return _fields_metadata_dictionary.FieldsMetadataDictionary.create(fields);
  }
  static create({
    otelFields
  }) {
    // Convert structured semconv fields to FieldMetadataPlain format
    const convertedFields = convertSemconvToFieldMetadata(otelFields);
    const decodedFields = _fields_metadata.fieldsMetadataDictionaryRT.decode(convertedFields);
    if ((0, _Either.isLeft)(decodedFields)) {
      throw Error(`OtelFieldsRepositoryDeps.create: could not validate data: ${_PathReporter.PathReporter.report(decodedFields).join('\n')}`);
    }
    return new OtelFieldsRepository(decodedFields.right);
  }
}
exports.OtelFieldsRepository = OtelFieldsRepository;