"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.list = exports.byIndex = exports.byField = void 0;
var _walker = require("../../../walker");
var _pretty_print = require("../../../pretty_print");
var _builder = require("../../../builder");
var generic = _interopRequireWildcard(require("../../generic"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Lists all "WHERE" commands in the query AST.
 *
 * @param ast The root AST node to search for "WHERE" commands.
 * @returns A collection of "WHERE" commands.
 */
const list = ast => {
  return generic.commands.list(ast, cmd => cmd.name === 'where');
};

/**
 * Retrieves the "WHERE" command at the specified index in order of appearance.
 *
 * @param ast The root AST node to search for "WHERE" commands.
 * @param index The index of the "WHERE" command to retrieve.
 * @returns The "WHERE" command at the specified index, if any.
 */
exports.list = list;
const byIndex = (ast, index) => {
  return [...list(ast)][index];
};
exports.byIndex = byIndex;
const fieldTemplateToField = template => {
  if (typeof template === 'string') {
    const part = template.startsWith('?') ? _builder.Builder.param.build(template) : _builder.Builder.identifier({
      name: template
    });
    const column = _builder.Builder.expression.column({
      args: [part]
    });
    return column;
  } else if (Array.isArray(template)) {
    const identifiers = template.map(name => {
      if (name.startsWith('?')) {
        return _builder.Builder.param.build(name);
      } else {
        return _builder.Builder.identifier({
          name
        });
      }
    });
    const column = _builder.Builder.expression.column({
      args: identifiers
    });
    return column;
  }
  return template;
};
const matchNodeAgainstField = (node, field) => {
  return _pretty_print.LeafPrinter.print(node) === _pretty_print.LeafPrinter.print(field);
};

/**
 * Finds the first "WHERE" command which contains the specified text as one of
 * its comparison operands. The text can represent a field (including nested
 * fields or a single identifier), or a param. If the text starts with "?", it
 * is assumed to be a param.
 *
 * Examples:
 *
 * ```ts
 * byField(ast, 'field');
 * byField(ast, ['nested', 'field']);
 * byField(ast, '?param');
 * byField(ast, ['nested', '?param']);
 * byField(ast, ['nested', 'positional', 'param', '?123']);
 * byField(ast, '?');
 * ```
 *
 * Alternatively you can build your own field template using the builder:
 *
 * ```ts
 * byField(ast, Builder.expression.column({
 *   args: [Builder.identifier({ name: 'field' })]
 * }));
 * ```
 *
 * @param ast The root AST node search for "WHERE" commands.
 * @param text The text or nested column name texts to search for.
 */
const byField = (ast, template) => {
  const field = fieldTemplateToField(template);
  for (const command of list(ast)) {
    let found;
    const matchNode = node => {
      if (found) {
        return;
      }
      if (matchNodeAgainstField(node, field)) {
        found = node;
      }
    };
    _walker.Walker.walk(command, {
      visitColumn: matchNode,
      visitIdentifier: matchNode,
      visitLiteral: matchNode
    });
    if (found) {
      return [command, found];
    }
  }
  return undefined;
};
exports.byField = byField;