"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upsert = exports.removeByPredicate = exports.remove = exports.list = exports.insert = exports.find = void 0;
var _walker = require("../../../walker");
var _visitor = require("../../../visitor");
var _util = require("../../util");
var generic = _interopRequireWildcard(require("../../generic"));
var _builder = require("../../../builder");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Returns all METADATA field AST nodes and their corresponding parent command
 * option nodes.
 *
 * @param ast The root AST node to search for metadata fields.
 * @returns A collection of [column, option] pairs for each metadata field found.
 */
const list = ast => {
  const visitor = new _visitor.Visitor().on('visitExpression', function* () {}).on('visitColumnExpression', function* (ctx) {
    yield ctx.node;
  }).on('visitCommandOption', function* (ctx) {
    if (ctx.node.name !== 'metadata') {
      return;
    }
    for (const args of ctx.visitArguments()) {
      for (const column of args) {
        yield [column, ctx.node];
      }
    }
  }).on('visitFromCommand', function* (ctx) {
    for (const options of ctx.visitOptions()) {
      yield* options;
    }
  }).on('visitCommand', function* () {}).on('visitQuery', function* (ctx) {
    for (const command of ctx.visitCommands()) {
      yield* command;
    }
  });
  if (ast.type === 'command') {
    return visitor.visitCommand(ast);
  } else {
    return visitor.visitQuery(ast);
  }
};

/**
 * Find a METADATA field by its name or parts.
 *
 * @param ast The root AST node to search for metadata fields.
 * @param fieldName The name or parts of the field to find.
 * @returns A 2-tuple containing the column and the option it was found in, or
 *     `undefined` if the field was not found.
 */
exports.list = list;
const find = (ast, fieldName) => {
  if (typeof fieldName === 'string') {
    fieldName = [fieldName];
  }
  const predicate = ([field]) => (0, _util.cmpArr)(field.args.map(arg => arg.type === 'identifier' ? arg.name : ''), fieldName);
  return (0, _util.findByPredicate)(list(ast), predicate);
};

/**
 * Removes the first found METADATA field that satisfies the predicate.
 *
 * @param ast The root AST node to search for metadata fields.
 * @param predicate The predicate function to filter fields.
 * @returns The removed column and option, if any.
 */
exports.find = find;
const removeByPredicate = (ast, predicate) => {
  const tuple = (0, _util.findByPredicate)(list(ast), ([field]) => predicate(field));
  if (!tuple) {
    return;
  }
  const [column, option] = tuple;
  const index = option.args.indexOf(column);
  if (index === -1) {
    return;
  }
  option.args.splice(index, 1);
  if (option.args.length === 0) {
    generic.commands.options.remove(ast, option);
  }
  return tuple;
};

/**
 * Removes the first METADATA field that matches the given name and returns
 * a 2-tuple (the column and the option it was removed from).
 *
 * @param ast The root AST node to search for metadata fields.
 * @param fieldName The name or parts of the field to remove.
 * @returns The removed column and option, if any.
 */
exports.removeByPredicate = removeByPredicate;
const remove = (ast, fieldName) => {
  if (typeof fieldName === 'string') {
    fieldName = [fieldName];
  }
  return removeByPredicate(ast, field => (0, _util.cmpArr)(field.args.map(arg => arg.type === 'identifier' ? arg.name : ''), fieldName));
};

/**
 * Insert into a specific position or append a `METADATA` field to the `FROM`
 * command.
 *
 * @param ast The root AST node.
 * @param fieldName Field name or parts as an array, e.g. `['a', 'b']`.
 * @param index Position to insert the field at. If `-1` or not specified, the
 *     field will be appended.
 * @returns If the field was successfully inserted, returns a 2-tuple containing
 *     the column and the option it was inserted into. Otherwise, returns
 *    `undefined`.
 */
exports.remove = remove;
const insert = (ast, fieldName, index = -1) => {
  let option = generic.commands.options.findByName(ast, 'from', 'metadata');
  if (!option) {
    const command = generic.commands.findByName(ast, 'from');
    if (!command) {
      return;
    }
    option = generic.commands.options.append(command, 'metadata');
  }
  const parts = typeof fieldName === 'string' ? [fieldName] : fieldName;
  const args = parts.map(part => _builder.Builder.identifier({
    name: part
  }));
  const column = _builder.Builder.expression.column({
    args
  });
  if (index === -1) {
    option.args.push(column);
  } else {
    option.args.splice(index, 0, column);
  }
  return [column, option];
};

/**
 * The `.upsert()` method works like `.insert()`, but will not insert a field
 * if it already exists.
 *
 * @param ast The root AST node.
 * @param fieldName The field name or parts as an array, e.g. `['a', 'b']`.
 * @param index Position to insert the field at. If `-1` or not specified, the
 *     field will be appended.
 * @returns If the field was successfully inserted, returns a 2-tuple containing
 *     the column and the option it was inserted into. Otherwise, returns
 *    `undefined`.
 */
exports.insert = insert;
const upsert = (ast, fieldName, index = -1) => {
  const option = generic.commands.options.findByName(ast, 'from', 'metadata');
  if (option) {
    const parts = Array.isArray(fieldName) ? fieldName : [fieldName];
    const existing = _walker.Walker.find(option, node => node.type === 'column' && (0, _util.cmpArr)(node.args.map(arg => arg.type === 'identifier' ? arg.name : ''), parts));
    if (existing) {
      return undefined;
    }
  }
  return insert(ast, fieldName, index);
};
exports.upsert = upsert;