"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.columnsAfter = void 0;
exports.extractSemanticsFromGrok = extractSemanticsFromGrok;
var _lodash = require("lodash");
var _walker = require("../../../walker");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function unquoteTemplate(inputString) {
  if (inputString.startsWith('"') && inputString.endsWith('"') && inputString.length >= 2) {
    return inputString.substring(1, inputString.length - 1);
  }
  return inputString;
}
function extractSemanticsFromGrok(pattern) {
  const columns = [];

  // Regex for Grok's %{SYNTAX:SEMANTIC:TYPE} pattern
  const grokSyntaxRegex = /%{\w+:(?<column>[\w@]+)(?::(?<type>\w+))?}/g;
  let grokMatch;
  while ((grokMatch = grokSyntaxRegex.exec(pattern)) !== null) {
    var _grokMatch, _grokMatch$groups;
    if ((_grokMatch = grokMatch) !== null && _grokMatch !== void 0 && (_grokMatch$groups = _grokMatch.groups) !== null && _grokMatch$groups !== void 0 && _grokMatch$groups.column) {
      columns.push({
        name: grokMatch.groups.column,
        type: grokMatch.groups.type ? grokTypeToESQLFieldType(grokMatch.groups.type) : 'keyword'
      });
    }
  }

  // Regex for Oniguruma-style named capture groups (?<name>...) or (?'name'...)
  // Oniguruma supports both `?<name>` and `?'name'` for named capture groups.
  const onigurumaNamedCaptureRegex = /(?<column>\(\?<(\w+)>|\(\?'(\w+)'\)[^)]*\))/g;
  let onigurumaMatch;
  while ((onigurumaMatch = onigurumaNamedCaptureRegex.exec(pattern)) !== null) {
    // If it's a (?<name>...) style
    if (onigurumaMatch[2]) {
      columns.push({
        name: onigurumaMatch[2],
        type: 'keyword'
      });
    }
    // If it's a (?'name'...) style
    else if (onigurumaMatch[3]) {
      columns.push({
        name: onigurumaMatch[3],
        type: 'keyword'
      });
    }
  }

  // Remove duplicates by name
  return (0, _lodash.uniqBy)(columns, 'name');
}
const columnsAfter = (command, previousColumns, _query) => {
  const columns = [];
  (0, _walker.walk)(command, {
    visitLiteral: node => {
      const grokRegex = unquoteTemplate(String(node.value));
      columns.push(...extractSemanticsFromGrok(grokRegex));
    }
  });
  return [...previousColumns, ...columns.map(column => ({
    name: column.name,
    type: column.type,
    userDefined: false
  }))];
};

/**
 * Maps Grok data types to ES|QL field types.
 * Reference: https://www.elastic.co/guide/en/elasticsearch/reference/current/grok-processor.html
 */
exports.columnsAfter = columnsAfter;
const GROK_TO_ESQL_TYPE_MAP = {
  int: 'integer',
  long: 'long',
  float: 'double',
  double: 'double',
  boolean: 'boolean'
};
function grokTypeToESQLFieldType(grokType) {
  var _GROK_TO_ESQL_TYPE_MA;
  const normalizedType = grokType.toLowerCase();
  return (_GROK_TO_ESQL_TYPE_MA = GROK_TO_ESQL_TYPE_MAP[normalizedType]) !== null && _GROK_TO_ESQL_TYPE_MA !== void 0 ? _GROK_TO_ESQL_TYPE_MA : 'keyword';
}