"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autocomplete = autocomplete;
var _helpers = require("../../../definitions/utils/autocomplete/helpers");
var _functions = require("../../../definitions/utils/functions");
var _operators = require("../../../definitions/utils/operators");
var _shared = require("../../../definitions/utils/shared");
var _complete_items = require("../../complete_items");
var _types = require("../../types");
var _util = require("./util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function autocomplete(query, command, callbacks, context, cursorPosition) {
  var _context$policies, _context$columns, _command$args$find;
  const innerText = query.substring(0, cursorPosition);
  const pos = (0, _util.getPosition)(innerText, command);
  const policies = (_context$policies = context === null || context === void 0 ? void 0 : context.policies) !== null && _context$policies !== void 0 ? _context$policies : new Map();
  const columnMap = (_context$columns = context === null || context === void 0 ? void 0 : context.columns) !== null && _context$columns !== void 0 ? _context$columns : new Map();
  const fieldNames = [];
  for (const name of columnMap.keys()) {
    const col = columnMap.get(name);
    if (col && !col.userDefined) fieldNames.push(name);
  }
  const policyName = (_command$args$find = command.args.find(arg => !Array.isArray(arg) && arg.type === 'source')) === null || _command$args$find === void 0 ? void 0 : _command$args$find.name;
  const getFieldSuggestionsForWithClause = async () => {
    if (!policyName) {
      return [];
    }
    const policyMetadata = (0, _util.getPolicyMetadata)(policies, policyName);
    if (!policyMetadata) {
      return [];
    }
    const fieldSuggestions = (0, _functions.buildFieldsDefinitions)(policyMetadata.enrichFields, false);
    const lastWord = (0, _helpers.findFinalWord)(innerText);
    if (lastWord) {
      // ENRICH ... WITH a <suggest>
      const rangeToReplace = {
        start: innerText.length - lastWord.length + 1,
        end: innerText.length + 1
      };
      fieldSuggestions.forEach(s => {
        s.rangeToReplace = rangeToReplace;
      });
    }
    return fieldSuggestions;
  };
  switch (pos) {
    case _util.Position.MODE:
      return _util.modeSuggestions;
    case _util.Position.POLICY:
      {
        const policiesSuggestions = (0, _util.buildPoliciesDefinitions)(Array.from(policies.values()));
        const lastWord = (0, _helpers.findFinalWord)(innerText);
        if (lastWord !== '') {
          policiesSuggestions.forEach(policySuggestion => {
            policySuggestion.rangeToReplace = {
              start: innerText.length - lastWord.length + 1,
              end: innerText.length + 1
            };
          });
        }
        return policiesSuggestions.length ? policiesSuggestions : [_util.noPoliciesAvailableSuggestion];
      }
    case _util.Position.AFTER_POLICY:
      return [_util.onSuggestion, _util.withSuggestion, _complete_items.pipeCompleteItem];
    case _util.Position.MATCH_FIELD:
      {
        if (!policyName) {
          return [];
        }
        const policyMetadata = (0, _util.getPolicyMetadata)(policies, policyName);
        if (!policyMetadata) {
          return [];
        }
        return (0, _util.buildMatchingFieldsDefinition)(policyMetadata.matchField, fieldNames);
      }
    case _util.Position.AFTER_ON_CLAUSE:
      return [_util.withSuggestion, _complete_items.pipeCompleteItem];
    case _util.Position.WITH_NEW_CLAUSE:
      {
        var _callbacks$getSuggest;
        if (!policyName) {
          return [];
        }
        const policyMetadata = (0, _util.getPolicyMetadata)(policies, policyName);
        if (!policyMetadata) {
          return [];
        }
        const suggestions = [];
        suggestions.push((0, _complete_items.getNewUserDefinedColumnSuggestion)((callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getSuggest = callbacks.getSuggestedUserDefinedColumnName) === null || _callbacks$getSuggest === void 0 ? void 0 : _callbacks$getSuggest.call(callbacks, policyMetadata.enrichFields)) || ''));
        suggestions.push(...(await getFieldSuggestionsForWithClause()));
        return suggestions;
      }
    case _util.Position.WITH_AFTER_FIRST_WORD:
      {
        if (!policyName) {
          return [];
        }
        const policyMetadata = (0, _util.getPolicyMetadata)(policies, policyName);
        if (!policyMetadata) {
          return [];
        }
        const word = (0, _helpers.findPreviousWord)(innerText);
        if (policyMetadata.enrichFields.includes((0, _shared.unescapeColumnName)(word))) {
          // complete field name
          return [_complete_items.pipeCompleteItem, (0, _helpers.withAutoSuggest)(_complete_items.commaCompleteItem)];
        } else {
          // not recognized as a field name, assume new user-defined column name
          return (0, _operators.getOperatorSuggestions)({
            location: _types.Location.ENRICH
          }, callbacks === null || callbacks === void 0 ? void 0 : callbacks.hasMinimumLicenseRequired, context === null || context === void 0 ? void 0 : context.activeProduct);
        }
      }
    case _util.Position.WITH_AFTER_ASSIGNMENT:
      {
        const suggestions = [];
        suggestions.push(...(await getFieldSuggestionsForWithClause()));
        return suggestions;
      }
    case _util.Position.WITH_AFTER_COMPLETE_CLAUSE:
      {
        return [_complete_items.pipeCompleteItem, (0, _helpers.withAutoSuggest)(_complete_items.commaCompleteItem)];
      }
    default:
      return [];
  }
}