"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findEntityDefinitionById = findEntityDefinitionById;
exports.findEntityDefinitions = findEntityDefinitions;
var _lodash = require("lodash");
var _saved_objects = require("../../saved_objects");
var _built_in = require("./built_in");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function findEntityDefinitions({
  soClient,
  esClient,
  builtIn,
  id,
  page = 1,
  perPage = 10,
  includeState = false,
  type
}) {
  const filter = (0, _lodash.compact)([typeof builtIn === 'boolean' ? `${_saved_objects.SO_ENTITY_DEFINITION_TYPE}.attributes.id:(${_built_in.BUILT_IN_ID_PREFIX}*)` : undefined, id ? `${_saved_objects.SO_ENTITY_DEFINITION_TYPE}.attributes.id:(${id})` : undefined, type ? `${_saved_objects.SO_ENTITY_DEFINITION_TYPE}.attributes.type:(${type})` : undefined]).join(' AND ');
  const response = await soClient.find({
    type: _saved_objects.SO_ENTITY_DEFINITION_TYPE,
    filter,
    page,
    perPage
  });
  if (!includeState) {
    return response.saved_objects.map(({
      attributes
    }) => attributes);
  }
  return Promise.all(response.saved_objects.map(async ({
    attributes
  }) => {
    const state = await getEntityDefinitionState(esClient, attributes);
    return {
      ...attributes,
      state
    };
  }));
}
async function findEntityDefinitionById({
  id,
  esClient,
  soClient,
  includeState = false
}) {
  const [definition] = await findEntityDefinitions({
    esClient,
    soClient,
    id,
    includeState,
    perPage: 1
  });
  return definition;
}
async function getEntityDefinitionState(esClient, definition) {
  const [ingestPipelines, transforms, indexTemplates] = await Promise.all([getIngestPipelineState({
    definition,
    esClient
  }), getTransformState({
    definition,
    esClient
  }), getIndexTemplatesState({
    definition,
    esClient
  })]);
  const installed = ingestPipelines.every(pipeline => pipeline.installed) && transforms.every(transform => transform.installed) && indexTemplates.every(template => template.installed);
  const running = transforms.every(transform => transform.running);
  return {
    installed,
    running,
    components: {
      transforms,
      ingestPipelines,
      indexTemplates
    }
  };
}
async function getTransformState({
  definition,
  esClient
}) {
  var _definition$installed;
  const transformIds = ((_definition$installed = definition.installedComponents) !== null && _definition$installed !== void 0 ? _definition$installed : []).filter(({
    type
  }) => type === 'transform').map(({
    id
  }) => id);
  const transformStats = await Promise.all(transformIds.map(id => esClient.transform.getTransformStats({
    transform_id: id
  }))).then(results => results.map(({
    transforms
  }) => transforms).flat());
  return transformIds.map(id => {
    const stats = transformStats.find(transform => transform.id === id);
    if (!stats) {
      return {
        id,
        installed: false,
        running: false
      };
    }
    return {
      id,
      installed: true,
      running: stats.state === 'started' || stats.state === 'indexing',
      stats
    };
  });
}
async function getIngestPipelineState({
  definition,
  esClient
}) {
  var _definition$installed2;
  const ingestPipelineIds = ((_definition$installed2 = definition.installedComponents) !== null && _definition$installed2 !== void 0 ? _definition$installed2 : []).filter(({
    type
  }) => type === 'ingest_pipeline').map(({
    id
  }) => id);
  const ingestPipelines = await esClient.ingest.getPipeline({
    id: ingestPipelineIds.join(',')
  }, {
    ignore: [404]
  });
  return ingestPipelineIds.map(id => ({
    id,
    installed: !!ingestPipelines[id]
  }));
}
async function getIndexTemplatesState({
  definition,
  esClient
}) {
  var _definition$installed3;
  const indexTemplatesIds = ((_definition$installed3 = definition.installedComponents) !== null && _definition$installed3 !== void 0 ? _definition$installed3 : []).filter(({
    type
  }) => type === 'template').map(({
    id
  }) => id);
  const templates = await Promise.all(indexTemplatesIds.map(id => esClient.indices.getIndexTemplate({
    name: id
  }, {
    ignore: [404]
  }).then(({
    index_templates: indexTemplates
  }) => indexTemplates === null || indexTemplates === void 0 ? void 0 : indexTemplates[0]))).then(_lodash.compact);
  return indexTemplatesIds.map(id => {
    const template = templates.find(({
      name
    }) => name === id);
    if (!template) {
      return {
        id,
        installed: false
      };
    }
    return {
      id,
      installed: true,
      stats: template.index_template
    };
  });
}