"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAttackDiscoveryGenerationRoute = exports.PER_PAGE = void 0;
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _actions = require("@kbn/security-solution-features/actions");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _helpers = require("../../../helpers");
var _throw_if_public_api_disabled = require("../../helpers/throw_if_public_api_disabled");
var _build_response = require("../../../../lib/build_response");
var _get_generation = require("./helpers/get_generation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PER_PAGE = exports.PER_PAGE = 1000; // max 1000 discoveries for a single generation

const getAttackDiscoveryGenerationRoute = router => {
  router.versioned.get({
    access: 'public',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_GENERATIONS_BY_ID,
    security: {
      authz: {
        requiredPrivileges: [_actions.ATTACK_DISCOVERY_API_ACTION_ALL]
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.GetAttackDiscoveryGenerationRequestQuery),
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.GetAttackDiscoveryGenerationRequestParams)
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user checks:
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }
    try {
      var _request$query$enable, _request$query, _request$query$with_r, _request$query2;
      await (0, _throw_if_public_api_disabled.throwIfPublicApiDisabled)(context);
      const {
        execution_uuid: executionUuid
      } = request.params;
      const enableFieldRendering = (_request$query$enable = (_request$query = request.query) === null || _request$query === void 0 ? void 0 : _request$query.enable_field_rendering) !== null && _request$query$enable !== void 0 ? _request$query$enable : false; // public APIs default to NOT rendering fields as a convenience to non-Kibana clients
      const withReplacements = (_request$query$with_r = (_request$query2 = request.query) === null || _request$query2 === void 0 ? void 0 : _request$query2.with_replacements) !== null && _request$query$with_r !== void 0 ? _request$query$with_r : true; // public APIs default to applying replacements in responses as a convenience to non-Kibana clients
      const eventLogIndex = (await context.elasticAssistant).eventLogIndex;
      const spaceId = (await context.elasticAssistant).getSpaceId();
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const currentUser = await checkResponse.currentUser;
      const findResponse = await dataClient.findAttackDiscoveryAlerts({
        authenticatedUser: currentUser,
        esClient: ctx.core.elasticsearch.client.asCurrentUser,
        findAttackDiscoveryAlertsParams: {
          enableFieldRendering,
          executionUuid,
          page: 1,
          perPage: PER_PAGE,
          sortField: '@timestamp',
          withReplacements
        },
        logger
      });
      const data = findResponse.data;
      const generation = await (0, _get_generation.getGeneration)({
        dataClient,
        authenticatedUser: currentUser,
        eventLogIndex,
        executionUuid,
        logger,
        spaceId,
        data
      });
      const responseBody = {
        generation,
        data
      };
      return response.ok({
        body: responseBody
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.getAttackDiscoveryGenerationRoute = getAttackDiscoveryGenerationRoute;