"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadDefendInsights = exports.getDefendInsightsDocsCount = void 0;
var _fs = require("fs");
var _directory = require("langchain/document_loaders/fs/directory");
var _text = require("langchain/document_loaders/fs/text");
var _path = require("path");
var _pMap = _interopRequireDefault(require("p-map"));
var _normalizePath = _interopRequireDefault(require("normalize-path"));
var _constants = require("../../../routes/knowledge_base/constants");
var _app_context = require("../../../services/app_context");
var _helpers = require("../../../routes/helpers");
var _add_required_kb_resource_metadata = require("./add_required_kb_resource_metadata");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Loads Defend Insights knowledge base files organized by insight type
 */
const loadDefendInsights = async (kbDataClient, logger) => {
  const isPolicyResponseFailureEnabled = _app_context.appContextService.getRegisteredFeatures(_helpers.DEFAULT_PLUGIN_NAME).defendInsightsPolicyResponseFailure;
  if (!isPolicyResponseFailureEnabled) {
    return true;
  }
  try {
    const subdirectories = [{
      path: 'policy_response_failure',
      resource: _constants.DEFEND_INSIGHTS_POLICY_RESPONSE_FAILURE
    }];
    let totalDocsLoaded = 0;
    for (const {
      path: subPath,
      resource
    } of subdirectories) {
      const docsLoader = new _directory.DirectoryLoader((0, _path.resolve)(__dirname, `../../../knowledge_base/defend_insights/${subPath}`), {
        '.md': path => new _text.TextLoader(path)
      }, true);
      const rawDocs = await docsLoader.load();
      if (rawDocs.length > 0) {
        var _response$length;
        // Add metadata specific to this resource type
        const docs = (0, _add_required_kb_resource_metadata.addRequiredKbResourceMetadata)({
          docs: rawDocs,
          kbResource: resource,
          required: true // Mark as required for defend insights
        });
        logger.info(`Loading ${docs.length} Defend Insights docs from ${subPath} into the Knowledge Base`);

        // Ingest documents one by one to avoid blocking Inference Endpoint
        const response = (await (0, _pMap.default)(docs, singleDoc => kbDataClient.addKnowledgeBaseDocuments({
          documents: [singleDoc],
          global: true
        }), {
          concurrency: 1
        })).flat();
        const loadedCount = (_response$length = response === null || response === void 0 ? void 0 : response.length) !== null && _response$length !== void 0 ? _response$length : 0;
        totalDocsLoaded += loadedCount;
        logger.info(`Loaded ${loadedCount} Defend Insights docs from ${subPath}`);
      } else {
        logger.info(`No documents found in defend_insights/${subPath}`);
      }
    }
    logger.info(`Total Defend Insights docs loaded: ${totalDocsLoaded}`);
    return totalDocsLoaded > 0;
  } catch (e) {
    logger.error(`Failed to load Defend Insights docs into the Knowledge Base\n${e}`);
    return false;
  }
};
exports.loadDefendInsights = loadDefendInsights;
const getDefendInsightsDocsCount = async ({
  logger
}) => {
  try {
    const files = (0, _fs.globSync)('**/*.{md,txt}', {
      cwd: (0, _path.resolve)(__dirname, '../../../knowledge_base/defend_insights')
    }).map(p => (0, _normalizePath.default)(p));
    return files.length;
  } catch (e) {
    logger.error(`Failed to get Defend Insights source docs count\n${e}`);
    return 0;
  }
};
exports.getDefendInsightsDocsCount = getDefendInsightsDocsCount;