"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.attackDiscoveryScheduleExecutor = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/alerting-plugin/server");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _telemetry = require("../../../../routes/attack_discovery/helpers/telemetry");
var _constants = require("../../../../ai_assistant_service/constants");
var _helpers = require("../../../../ai_assistant_data_clients/anonymization_fields/helpers");
var _ai_assistant_service = require("../../../../ai_assistant_service");
var _find = require("../../../../ai_assistant_data_clients/find");
var _generate_discoveries = require("../../../../routes/attack_discovery/helpers/generate_discoveries");
var _helpers2 = require("../../../data_stream/helpers");
var _transform_to_alert_documents = require("../../persistence/transforms/transform_to_alert_documents");
var _deduplication = require("../../persistence/deduplication");
var _get_scheduled_index_pattern = require("../../persistence/get_scheduled_index_pattern");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const attackDiscoveryScheduleExecutor = async ({
  options,
  logger,
  publicBaseUrl,
  telemetry
}) => {
  const {
    params,
    rule,
    services,
    spaceId
  } = options;
  const {
    alertsClient,
    actionsClient,
    savedObjectsClient,
    scopedClusterClient
  } = services;
  if (!alertsClient) {
    throw new _server.AlertsClientError();
  }
  if (!actionsClient) {
    throw new Error('Expected actionsClient not to be null!');
  }
  const esClient = scopedClusterClient.asCurrentUser;
  const resourceName = (0, _ai_assistant_service.getResourceName)(_constants.ANONYMIZATION_FIELDS_RESOURCE);
  const index = (0, _helpers2.getIndexTemplateAndPattern)(resourceName, spaceId).alias;
  const result = await (0, _find.findDocuments)({
    esClient,
    page: 1,
    perPage: 1000,
    index,
    logger
  });
  const anonymizationFields = (0, _helpers.transformESSearchToAnonymizationFields)(result.data);
  const {
    query,
    filters,
    combinedFilter,
    ...restParams
  } = params;
  const startTime = (0, _moment.default)(); // start timing the generation
  const scheduleInfo = {
    id: rule.id,
    interval: rule.schedule.interval,
    actions: rule.actions.map(({
      actionTypeId
    }) => actionTypeId)
  };
  try {
    const {
      anonymizedAlerts,
      attackDiscoveries,
      replacements
    } = await (0, _generate_discoveries.generateAttackDiscoveries)({
      actionsClient,
      config: {
        ...restParams,
        filter: combinedFilter,
        anonymizationFields,
        subAction: 'invokeAI'
      },
      esClient,
      logger,
      savedObjectsClient
    });

    // Remove this when alerting framework adds a way to abort rule execution:
    // https://github.com/elastic/kibana/issues/219152
    if (services.shouldStopExecution()) {
      throw new Error('Rule execution cancelled due to timeout');
    }
    const endTime = (0, _moment.default)();
    const durationMs = endTime.diff(startTime);
    (0, _telemetry.reportAttackDiscoveryGenerationSuccess)({
      alertsContextCount: anonymizedAlerts.length,
      apiConfig: params.apiConfig,
      attackDiscoveries,
      durationMs,
      end: restParams.end,
      hasFilter: !!(combinedFilter && Object.keys(combinedFilter).length),
      scheduleInfo,
      size: restParams.size,
      start: restParams.start,
      telemetry
    });
    const alertsParams = {
      alertsContextCount: anonymizedAlerts.length,
      anonymizedAlerts,
      apiConfig: params.apiConfig,
      connectorName: params.apiConfig.name,
      enableFieldRendering: true,
      // Always enable field rendering for scheduled discoveries. It's still possible for clients who read the generated discoveries to specify false when retrieving them.
      replacements,
      withReplacements: false // Never apply replacements to the results. It's still possible for clients who read the generated discoveries to specify true when retrieving them.
    };

    // Deduplicate attackDiscoveries before creating alerts
    const indexPattern = (0, _get_scheduled_index_pattern.getScheduledIndexPattern)(spaceId);
    const dedupedDiscoveries = await (0, _deduplication.deduplicateAttackDiscoveries)({
      esClient,
      attackDiscoveries: attackDiscoveries !== null && attackDiscoveries !== void 0 ? attackDiscoveries : [],
      connectorId: params.apiConfig.connectorId,
      indexPattern,
      logger,
      ownerInfo: {
        id: rule.id,
        isSchedule: true
      },
      replacements,
      spaceId
    });
    await Promise.all(dedupedDiscoveries.map(async attackDiscovery => {
      const alertInstanceId = (0, _transform_to_alert_documents.generateAttackDiscoveryAlertHash)({
        attackDiscovery,
        connectorId: params.apiConfig.connectorId,
        ownerId: rule.id,
        replacements,
        spaceId
      });
      const {
        uuid: alertDocId
      } = alertsClient.report({
        id: alertInstanceId,
        actionGroup: 'default'
      });
      const baseAlertDocument = (0, _transform_to_alert_documents.transformToBaseAlertDocument)({
        alertDocId,
        alertInstanceId,
        attackDiscovery,
        alertsParams,
        publicBaseUrl,
        spaceId
      });
      const {
        alertIds,
        timestamp,
        mitreAttackTactics
      } = attackDiscovery;
      const {
        detailsMarkdown,
        entitySummaryMarkdown,
        title,
        summaryMarkdown
      } = (0, _elasticAssistantCommon.getAttackDiscoveryMarkdownFields)({
        attackDiscovery,
        replacements
      });
      const context = {
        attack: {
          alertIds,
          detailsMarkdown,
          detailsUrl: baseAlertDocument[_ruleDataUtils.ALERT_URL],
          entitySummaryMarkdown,
          mitreAttackTactics,
          summaryMarkdown,
          timestamp,
          title
        }
      };
      alertsClient.setAlertData({
        id: alertInstanceId,
        payload: baseAlertDocument,
        context
      });
    }));
  } catch (error) {
    logger.error(error);
    const transformedError = (0, _securitysolutionEsUtils.transformError)(error);
    (0, _telemetry.reportAttackDiscoveryGenerationFailure)({
      apiConfig: params.apiConfig,
      errorMessage: transformedError.message,
      scheduleInfo,
      telemetry
    });
    throw error;
  }
  return {
    state: {}
  };
};
exports.attackDiscoveryScheduleExecutor = attackDiscoveryScheduleExecutor;