"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AIAssistantConversationsDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _create_conversation = require("./create_conversation");
var _update_conversation = require("./update_conversation");
var _get_conversation = require("./get_conversation");
var _conversation_exists = require("./conversation_exists");
var _delete_conversation = require("./delete_conversation");
var _append_conversation_messages = require("./append_conversation_messages");
var _ = require("..");
var _delete_all_conversations = require("./delete_all_conversations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Params for when creating ConversationDataClient in Request Context Factory. Useful if needing to modify
 * configuration after initial plugin start
 */

class AIAssistantConversationsDataClient extends _.AIAssistantDataClient {
  constructor(_options) {
    super(_options);
    /**
     * Checks if a conversation exists by ID without user access filtering.
     * @param options
     * @param options.id The conversation id to check.
     * @returns Promise<boolean> indicating whether the conversation exists
     */
    (0, _defineProperty2.default)(this, "conversationExists", async ({
      id
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _conversation_exists.conversationExists)({
        esClient,
        logger: this.options.logger,
        conversationIndex: this.indexTemplateAndPattern.alias,
        id
      });
    });
    /**
     * Gets a conversation by its id.
     * @param options
     * @param options.id The existing conversation id.
     * @param options.authenticatedUser Current authenticated user.
     * @returns The conversation response
     */
    (0, _defineProperty2.default)(this, "getConversation", async ({
      id,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _get_conversation.getConversation)({
        esClient,
        logger: this.options.logger,
        conversationIndex: this.indexTemplateAndPattern.alias,
        id,
        user: authenticatedUser !== null && authenticatedUser !== void 0 ? authenticatedUser : this.options.currentUser
      });
    });
    /**
     * Updates a conversation with the new messages.
     * @param options
     * @param options.conversation The existing conversation to which append the new messages.
     * @param options.messages Set this to true if this is a conversation that is "immutable"/"pre-packaged".
     * @returns The conversation updated
     */
    (0, _defineProperty2.default)(this, "appendConversationMessages", async ({
      existingConversation,
      messages,
      authenticatedUser
    }) => {
      var _ref;
      const dataWriter = await this.getWriter();
      return (0, _append_conversation_messages.appendConversationMessages)({
        dataWriter,
        logger: this.options.logger,
        existingConversation,
        messages,
        authenticatedUser: (_ref = authenticatedUser !== null && authenticatedUser !== void 0 ? authenticatedUser : this.options.currentUser) !== null && _ref !== void 0 ? _ref : undefined
      });
    });
    /**
     * Creates a conversation, if given at least the "title" and "g"
     * See {@link https://www.elastic.co/guide/en/security/current/}
     * for more information around formats of the deserializer and serializer
     * @param options
     * @param options.id The id of the conversation to create or "undefined" if you want an "id" to be auto-created for you
     * @param options.title A custom deserializer for the conversation. Optionally, you an define this as handle bars. See online docs for more information.
     * @param options.messages Set this to true if this is a conversation that is "immutable"/"pre-packaged".
     * @param options.g Determines how uploaded conversation item values are parsed. By default, conversation items are parsed using named regex groups. See online docs for more information.
     * @returns The conversation created
     */
    (0, _defineProperty2.default)(this, "createConversation", async ({
      conversation
    }) => {
      if (!this.options.currentUser) {
        throw new Error('AIAssistantConversationsDataClient currentUser is not defined.');
      }
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _create_conversation.createConversation)({
        esClient,
        logger: this.options.logger,
        conversationIndex: this.indexTemplateAndPattern.alias,
        spaceId: this.spaceId,
        user: this.options.currentUser,
        conversation
      });
    });
    /**
     * Updates a conversation container's value given the id of the conversation.
     * See {@link https://www.elastic.co/guide/en/elasticsearch/reference/current/optimistic-concurrency-control.html}
     * for more information around optimistic concurrency control.
     * @param options
     * @param options.conversationUpdateProps
     * @param options.conversationUpdateProps.id id of the conversation to replace the conversation container data with.
     * @param options.conversationUpdateProps.title The new tilet, or "undefined" if this should not be updated.
     * @param options.conversationUpdateProps.messages The new messages, or "undefined" if this should not be updated.
     * @param options.conversationUpdateProps.excludeFromLastConversationStorage The new value for excludeFromLastConversationStorage, or "undefined" if this should not be updated.
     * @param options.conversationUpdateProps.replacements The new value for replacements, or "undefined" if this should not be updated.
     */
    (0, _defineProperty2.default)(this, "updateConversation", async ({
      conversationUpdateProps,
      authenticatedUser
    }) => {
      var _ref2;
      const dataWriter = await this.getWriter();
      return (0, _update_conversation.updateConversation)({
        conversationUpdateProps,
        dataWriter,
        logger: this.options.logger,
        user: (_ref2 = authenticatedUser !== null && authenticatedUser !== void 0 ? authenticatedUser : this.options.currentUser) !== null && _ref2 !== void 0 ? _ref2 : undefined
      });
    });
    /**
     * Given a conversation id, this will delete the conversation from the id
     * @param options
     * @param options.id The id of the conversation to delete
     * @returns The conversation deleted if found, otherwise null
     */
    (0, _defineProperty2.default)(this, "deleteConversation", async id => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _delete_conversation.deleteConversation)({
        esClient,
        conversationIndex: this.indexTemplateAndPattern.alias,
        id,
        logger: this.options.logger
      });
    });
    /**
     * Deletes all conversations in the index.
     * @param options.excludedIds An array of ids to exclude from deletion.
     * @returns The number of conversations deleted
     */
    (0, _defineProperty2.default)(this, "deleteAllConversations", async options => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _delete_all_conversations.deleteAllConversations)({
        esClient,
        conversationIndex: this.indexTemplateAndPattern.alias,
        logger: this.options.logger,
        excludedIds: options === null || options === void 0 ? void 0 : options.excludedIds
      });
    });
    this.options = _options;
  }
}
exports.AIAssistantConversationsDataClient = AIAssistantConversationsDataClient;