"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.replaceColumnsWithVariableDriven = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Replaces columns in savedSearch with variable-driven columns when they exist in ESQL variables
 * and the original columns are not present in the current request.
 */
const replaceColumnsWithVariableDriven = (savedSearchColumns, columnsMeta, esqlVariables, isEsql) => {
  if (!isEsql || !columnsMeta) {
    return savedSearchColumns !== null && savedSearchColumns !== void 0 ? savedSearchColumns : [];
  }
  const columnsFromRequest = Object.keys(columnsMeta);
  const columnDrivenByVariable = Object.entries(columnsMeta).find(([id, meta]) => {
    // check if the id exists in the esqlVariables value property
    return esqlVariables === null || esqlVariables === void 0 ? void 0 : esqlVariables.some(esqlVar => esqlVar.value === id);
  });
  if (!columnDrivenByVariable) {
    return savedSearchColumns !== null && savedSearchColumns !== void 0 ? savedSearchColumns : [];
  }

  // find the savedSearch.columns which doesn't exist in columnsFromRequest and replace it with the columnDrivenByVariable
  const variableDrivenColumnName = columnDrivenByVariable[0];
  const updatedColumns = (savedSearchColumns !== null && savedSearchColumns !== void 0 ? savedSearchColumns : []).map(columnName => {
    // If this column from savedSearch doesn't exist in the current request columns,
    // replace it with the variable-driven column
    if (!columnsFromRequest.includes(columnName)) {
      return variableDrivenColumnName;
    }
    return columnName;
  });

  // Remove duplicates and return
  return Array.from(new Set(updatedColumns));
};
exports.replaceColumnsWithVariableDriven = replaceColumnsWithVariableDriven;