"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.copySavedSearch = copySavedSearch;
exports.getSavedSearchContainer = getSavedSearchContainer;
var _rxjs = require("rxjs");
var _esQuery = require("@kbn/es-query");
var _update_saved_search = require("./utils/update_saved_search");
var _add_log = require("../../../utils/add_log");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Container for the saved search state, allowing to update the saved search
 * It centralizes functionality that was spread across the Discover main codebase
 * There are 2 hooks to access the state of the saved search in React components:
 * - useSavedSearch for the current state, that's updated on every relevant state change
 */

function getSavedSearchContainer({
  services,
  internalState,
  getCurrentTab
}) {
  const initialSavedSearch = services.savedSearch.getNew();
  const savedSearchInitial$ = new _rxjs.BehaviorSubject(initialSavedSearch);
  const savedSearchCurrent$ = new _rxjs.BehaviorSubject(copySavedSearch(initialSavedSearch));
  const set = savedSearch => {
    (0, _add_log.addLog)('[savedSearch] set', savedSearch);
    savedSearchCurrent$.next(savedSearch);
    savedSearchInitial$.next(copySavedSearch(savedSearch));
    return savedSearch;
  };
  const getState = () => savedSearchCurrent$.getValue();
  const getInitial$ = () => savedSearchInitial$;
  const getCurrent$ = () => savedSearchCurrent$;
  const getTitle = () => savedSearchCurrent$.getValue().title;
  const getId = () => savedSearchCurrent$.getValue().id;
  const initUrlTracking = () => {
    const subscription = savedSearchCurrent$.subscribe(savedSearch => {
      const dataView = savedSearch.searchSource.getField('index');
      if (!(dataView !== null && dataView !== void 0 && dataView.id)) {
        return;
      }
      const dataViewSupportsTracking =
      // Disable for ad hoc data views, since they can't be restored after a page refresh
      dataView.isPersisted() ||
      // Unless it's a default profile data view, which can be restored on refresh
      internalState.getState().defaultProfileAdHocDataViewIds.includes(dataView.id) ||
      // Or we're in ES|QL mode, in which case we don't care about the data view
      (0, _esQuery.isOfAggregateQueryType)(savedSearch.searchSource.getField('query'));
      const trackingEnabled = dataViewSupportsTracking || Boolean(savedSearch.id);
      services.urlTracker.setTrackingEnabled(trackingEnabled);
    });
    return () => {
      subscription.unsubscribe();
    };
  };
  const assignNextSavedSearch = ({
    nextSavedSearch
  }) => {
    savedSearchCurrent$.next(nextSavedSearch);
  };
  const update = ({
    nextDataView,
    nextState,
    useFilterAndQueryServices
  }) => {
    (0, _add_log.addLog)('[savedSearch] update', {
      nextDataView,
      nextState
    });
    const previousSavedSearch = getState();
    const dataView = nextDataView ? nextDataView : previousSavedSearch.searchSource.getField('index');
    const nextSavedSearch = (0, _update_saved_search.updateSavedSearch)({
      savedSearch: {
        ...previousSavedSearch
      },
      dataView,
      initialInternalState: undefined,
      appState: nextState || {},
      globalState: getCurrentTab().globalState,
      services,
      useFilterAndQueryServices
    });
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] update done', nextSavedSearch);
    return nextSavedSearch;
  };
  const updateTimeRange = () => {
    const previousSavedSearch = getState();
    if (!previousSavedSearch.timeRestore) {
      return;
    }
    const refreshInterval = services.timefilter.getRefreshInterval();
    const nextSavedSearch = {
      ...previousSavedSearch,
      timeRange: services.timefilter.getTime(),
      refreshInterval: {
        value: refreshInterval.value,
        pause: refreshInterval.pause
      }
    };
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] updateWithTimeRange done', nextSavedSearch);
  };
  const updateVisContext = ({
    nextVisContext
  }) => {
    const previousSavedSearch = getState();
    const nextSavedSearch = {
      ...previousSavedSearch,
      visContext: nextVisContext
    };
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] updateVisContext done', nextSavedSearch);
  };
  const updateControlState = ({
    nextControlState
  }) => {
    const previousSavedSearch = getState();
    const nextSavedSearch = {
      ...previousSavedSearch,
      controlGroupJson: JSON.stringify(nextControlState)
    };
    assignNextSavedSearch({
      nextSavedSearch
    });
    (0, _add_log.addLog)('[savedSearch] updateControlState done', nextSavedSearch);
  };
  return {
    initUrlTracking,
    getCurrent$,
    getId,
    getInitial$,
    getState,
    getTitle,
    set,
    assignNextSavedSearch: nextSavedSearch => assignNextSavedSearch({
      nextSavedSearch
    }),
    update,
    updateTimeRange,
    updateVisContext,
    updateControlState
  };
}

/**
 * Copies a saved search object, due to the stateful nature of searchSource it has to be copied with a dedicated function
 * @param savedSearch
 */
function copySavedSearch(savedSearch) {
  return {
    ...savedSearch,
    ...{
      searchSource: savedSearch.searchSource.createCopy()
    }
  };
}