"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startQueryPerformanceTracking = startQueryPerformanceTracking;
var _rxjs = require("rxjs");
var _ebtTools = require("@kbn/ebt-tools");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _kibana_services = require("../../services/kibana_services");
var _telemetry_constants = require("../../utils/telemetry_constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

let isFirstDashboardLoadOfSession = true;
const loadTypesMapping = {
  sessionFirstLoad: 0,
  // on first time the SO is loaded
  dashboardFirstLoad: 1,
  // on initial load navigating into it
  dashboardSubsequentLoad: 2 // on filter-refresh
};
function startQueryPerformanceTracking(dashboard, performanceState) {
  return dashboard.children$.pipe((0, _rxjs.skipWhile)(children => {
    // Don't track render-status when the dashboard is still adding embeddables.
    return Object.values(children).length !== dashboard.getPanelCount();
  }), (0, _rxjs.map)(children => {
    // Filter for embeddables which publish phase events
    const childPhaseEventTrackers = [];
    const values = Object.values(children);
    for (const child of values) {
      if ((0, _presentationPublishing.apiPublishesPhaseEvents)(child)) {
        childPhaseEventTrackers.push(child);
      }
    }
    return childPhaseEventTrackers;
  }), (0, _rxjs.switchMap)(children => {
    if (children.length === 0) {
      return (0, _rxjs.of)([]); // map to empty stream
    }
    // Map to new stream of phase-events for each embeddable
    return (0, _rxjs.combineLatest)(children.map(child => child.phase$));
  }), (0, _rxjs.map)(latestPhaseEvents => {
    // Map individual render-state of panels to global render-state.
    return latestPhaseEvents.some(phaseEvent => {
      return phaseEvent && phaseEvent.status !== 'rendered';
    });
  }), (0, _rxjs.startWith)(false), (0, _rxjs.pairwise)()).subscribe(([wasDashboardStillLoading, isDashboardStillLoading]) => {
    const panelCount = dashboard.getPanelCount();
    const now = performance.now();
    const loadType = isFirstDashboardLoadOfSession ? 'sessionFirstLoad' : performanceState.firstLoad ? 'dashboardFirstLoad' : 'dashboardSubsequentLoad';
    const queryHasStarted = !wasDashboardStillLoading && isDashboardStillLoading;
    const queryHasFinished = wasDashboardStillLoading && !isDashboardStillLoading;
    if (performanceState.firstLoad && (panelCount === 0 || queryHasFinished)) {
      /**
       * we consider the Dashboard creation to be finished when all the panels are loaded.
       */
      performanceState.creationEndTime = now;
      isFirstDashboardLoadOfSession = false;
      performanceState.firstLoad = false;
    }
    if (queryHasStarted) {
      performanceState.lastLoadStartTime = now;
      return;
    }
    if (queryHasFinished) {
      var _performanceState$las, _performanceState$cre, _performanceState$cre2;
      const timeToData = now - ((_performanceState$las = performanceState.lastLoadStartTime) !== null && _performanceState$las !== void 0 ? _performanceState$las : now);
      const completeLoadDuration = ((_performanceState$cre = performanceState.creationEndTime) !== null && _performanceState$cre !== void 0 ? _performanceState$cre : now) - ((_performanceState$cre2 = performanceState.creationStartTime) !== null && _performanceState$cre2 !== void 0 ? _performanceState$cre2 : now);
      reportPerformanceMetrics({
        timeToData,
        panelCount,
        totalLoadTime: completeLoadDuration,
        loadType
      });
    }
  });
}
function reportPerformanceMetrics({
  timeToData,
  panelCount,
  totalLoadTime,
  loadType
}) {
  const duration = loadType === 'dashboardSubsequentLoad' ? timeToData : Math.max(timeToData, totalLoadTime);
  const meanPanelPrerender = (0, _ebtTools.getMeanFromPerformanceMeasures)({
    type: _ebtTools.PERFORMANCE_TRACKER_TYPES.PANEL,
    startMark: _ebtTools.PERFORMANCE_TRACKER_MARKS.PRE_RENDER,
    endMark: _ebtTools.PERFORMANCE_TRACKER_MARKS.RENDER_START
  });
  const meanPanelRenderComplete = (0, _ebtTools.getMeanFromPerformanceMeasures)({
    type: _ebtTools.PERFORMANCE_TRACKER_TYPES.PANEL,
    startMark: _ebtTools.PERFORMANCE_TRACKER_MARKS.RENDER_START,
    endMark: _ebtTools.PERFORMANCE_TRACKER_MARKS.RENDER_COMPLETE
  });
  const performanceMetricEvent = {
    eventName: _telemetry_constants.DASHBOARD_LOADED_EVENT,
    duration,
    key1: 'time_to_data',
    value1: timeToData,
    key2: 'num_of_panels',
    value2: panelCount,
    key4: 'load_type',
    value4: loadTypesMapping[loadType],
    key8: 'mean_panel_prerender',
    value8: meanPanelPrerender,
    key9: 'mean_panel_rendering',
    value9: meanPanelRenderComplete
  };
  (0, _ebtTools.reportPerformanceMetricEvent)(_kibana_services.coreServices.analytics, performanceMetricEvent);
  (0, _ebtTools.clearPerformanceTrackersByType)(_ebtTools.PERFORMANCE_TRACKER_TYPES.PANEL);
}