"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectSavedObjects = collectSavedObjects;
var _utils = require("@kbn/utils");
var _errors = require("../errors");
var _get_non_unique_entries = require("./get_non_unique_entries");
var _create_limit_stream = require("./create_limit_stream");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function collectSavedObjects({
  readStream,
  objectLimit,
  filter,
  supportedTypes,
  managed
}) {
  const errors = [];
  const entries = [];
  const importStateMap = new Map();
  const collectedObjects = await (0, _utils.createPromiseFromStreams)([readStream, (0, _create_limit_stream.createLimitStream)(objectLimit), (0, _utils.createFilterStream)(obj => {
    entries.push({
      type: obj.type,
      id: obj.id
    });
    if (supportedTypes.includes(obj.type)) {
      return true;
    }
    const {
      title
    } = obj.attributes;
    errors.push({
      id: obj.id,
      type: obj.type,
      meta: {
        title
      },
      error: {
        type: 'unsupported_type'
      }
    });
    return false;
  }), (0, _utils.createFilterStream)(obj => filter ? filter(obj) : true), (0, _utils.createMapStream)(obj => {
    var _ref;
    importStateMap.set(`${obj.type}:${obj.id}`, {});
    for (const ref of (_obj$references = obj.references) !== null && _obj$references !== void 0 ? _obj$references : []) {
      var _obj$references;
      const key = `${ref.type}:${ref.id}`;
      if (!importStateMap.has(key)) {
        importStateMap.set(key, {
          isOnlyReference: true
        });
      }
    }
    // Ensure migrations execute on every saved object
    return {
      ...obj,
      ...(!obj.migrationVersion && !obj.typeMigrationVersion ? {
        typeMigrationVersion: ''
      } : {}),
      // override any managed flag on an object with that given as an option otherwise set the default to avoid having to do that with a core migration transform
      // this is a bulk operation, applied to all objects being imported
      ...{
        managed: (_ref = managed !== null && managed !== void 0 ? managed : obj.managed) !== null && _ref !== void 0 ? _ref : false
      }
    };
  }), (0, _utils.createConcatStream)([])]);

  // throw a BadRequest error if we see the same import object type/id more than once
  const nonUniqueEntries = (0, _get_non_unique_entries.getNonUniqueEntries)(entries);
  if (nonUniqueEntries.length > 0) {
    throw _errors.SavedObjectsImportError.nonUniqueImportObjects(nonUniqueEntries);
  }
  return {
    errors,
    collectedObjects,
    importStateMap
  };
}