"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchSyncJobsStats = void 0;
var _searchConnectors = require("@kbn/search-connectors");
var _get_sync_jobs_queries = require("../../utils/get_sync_jobs_queries");
var _identify_exceptions = require("../../utils/identify_exceptions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchSyncJobsStats = async (client, isCrawler) => {
  try {
    const connectorIdsResult = await client.asCurrentUser.search({
      index: _searchConnectors.CONNECTORS_INDEX,
      scroll: '10s',
      stored_fields: []
    });
    const ids = connectorIdsResult.hits.hits.map(hit => hit._id);
    const orphanedJobsCountResponse = await client.asCurrentUser.count({
      index: _searchConnectors.CONNECTORS_JOBS_INDEX,
      query: (0, _get_sync_jobs_queries.getOrphanedJobsCountQuery)(ids, isCrawler)
    });
    const inProgressJobsCountResponse = await client.asCurrentUser.count({
      index: _searchConnectors.CONNECTORS_JOBS_INDEX,
      query: (0, _get_sync_jobs_queries.getInProgressJobsCountQuery)(isCrawler)
    });

    // Idle syncs don't make sense for Crawler, because it does not have concept of "Idle" syncs at all.
    // We tried tracking idle syncs in a way similar to connectors, but it results in all crawler jobs
    // marked as idle.
    const idleJobsCountResponse = isCrawler ? undefined : await client.asCurrentUser.count({
      index: _searchConnectors.CONNECTORS_JOBS_INDEX,
      query: (0, _get_sync_jobs_queries.getIdleJobsCountQuery)()
    });
    const errorResponse = await client.asCurrentUser.count({
      index: _searchConnectors.CONNECTORS_INDEX,
      query: (0, _get_sync_jobs_queries.getErrorCountQuery)(isCrawler)
    });
    const connectedResponse = await client.asCurrentUser.count({
      index: _searchConnectors.CONNECTORS_INDEX,
      query: (0, _get_sync_jobs_queries.getConnectedCountQuery)(isCrawler)
    });
    const incompleteResponse = await client.asCurrentUser.count({
      index: _searchConnectors.CONNECTORS_INDEX,
      query: (0, _get_sync_jobs_queries.getIncompleteCountQuery)(isCrawler)
    });
    const response = {
      connected: connectedResponse.count,
      errors: errorResponse.count,
      idle: (idleJobsCountResponse === null || idleJobsCountResponse === void 0 ? void 0 : idleJobsCountResponse.count) || 0,
      in_progress: inProgressJobsCountResponse.count,
      incomplete: incompleteResponse.count,
      orphaned_jobs: orphanedJobsCountResponse.count
    };
    return response;
  } catch (error) {
    if ((0, _identify_exceptions.isIndexNotFoundException)(error)) {
      return {
        connected: 0,
        errors: 0,
        idle: 0,
        in_progress: 0,
        incomplete: 0,
        orphaned_jobs: 0
      };
    }
    throw error;
  }
};
exports.fetchSyncJobsStats = fetchSyncJobsStats;