"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CasesConnectorExecutor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _jsonStableStringify = _interopRequireDefault(require("json-stable-stringify"));
var _pMap = _interopRequireDefault(require("p-map"));
var _lodash = require("lodash");
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _casesComponents = require("@kbn/cases-components");
var _std = require("@kbn/std");
var _constants = require("../../../common/constants");
var _common = require("../../../common");
var _constants2 = require("./constants");
var _utils = require("./utils");
var _cases_connector_error = require("./cases_connector_error");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CasesConnectorExecutor {
  constructor({
    logger,
    casesOracleService,
    casesService,
    casesClient,
    spaceId
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "casesOracleService", void 0);
    (0, _defineProperty2.default)(this, "casesService", void 0);
    (0, _defineProperty2.default)(this, "casesClient", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    (0, _defineProperty2.default)(this, "generateNoGroupAlertGrouping", groupingBy => {
      const noGroupedGrouping = groupingBy.reduce((acc, field) => {
        acc[field] = 'unknown';
        return acc;
      }, {});
      return noGroupedGrouping;
    });
    this.logger = logger;
    this.casesOracleService = casesOracleService;
    this.casesService = casesService;
    this.casesClient = casesClient;
    this.spaceId = spaceId;
  }
  async execute(params) {
    const {
      alerts,
      groupedAlerts: casesGroupedAlerts,
      groupingBy
    } = params;
    const groupedAlerts = casesGroupedAlerts !== null && casesGroupedAlerts !== void 0 ? casesGroupedAlerts : this.groupAlerts({
      params,
      alerts,
      groupingBy
    });
    const groupedAlertsWithCircuitBreakers = this.applyCircuitBreakers(params, groupedAlerts);
    if (groupedAlertsWithCircuitBreakers.length === 0) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor] Grouping did not produce any alerts. Skipping execution.`, this.getLogMetadata(params));
      return;
    }

    /**
     * Based on the rule ID, the grouping, the owner, the space ID,
     * the oracle record ID is generated
     */
    const groupedAlertsWithOracleKey = this.generateOracleKeys(params, groupedAlertsWithCircuitBreakers);

    /**
     * Gets all records by the IDs that produces in generateOracleKeys.
     * If a record does not exist it will create the record.
     * A record does not exist if it is the first time the connector run for a specific grouping.
     * The returned map will contain all records old and new.
     */
    const oracleRecordsMap = await this.upsertOracleRecords(params, groupedAlertsWithOracleKey);

    /**
     * If the time window has passed for a case we need to create a new case.
     * To do that we need to increase the record counter by one. Increasing the
     * counter will generate a new case ID for the same grouping.
     * The returned map contain all records with their counters updated correctly
     */
    const oracleRecordMapWithTimeWindowHandled = await this.handleTimeWindow(params, oracleRecordsMap);

    /**
     * Based on the rule ID, the grouping, the owner, the space ID,
     * and the counter of the oracle record the case ID is generated
     */
    const groupedAlertsWithCaseId = this.generateCaseIds(params, oracleRecordMapWithTimeWindowHandled);

    /**
     * Gets all records by the IDs that produces in generateCaseIds.
     * If a case does not exist it will create the case.
     * A case does not exist if it is the first time the connector run for a specific grouping
     * or the time window has elapsed and a new one should be created for the same grouping.
     * The returned map will contain all cases old and new.
     */
    const groupedAlertsWithCases = await this.upsertCases(params, groupedAlertsWithCaseId);

    /**
     * A user can configure how to handle closed cases. Based on the configuration
     * we open the closed cases by updating their status or we create new cases by
     * increasing the counter of the corresponding oracle record, generating the new
     * case ID, and creating the new case.
     * The map contains all cases updated and new without any remaining closed case.
     */
    const groupedAlertsWithClosedCasesHandled = await this.handleClosedCases(params, groupedAlertsWithCases);

    /**
     * Now that all cases are fetched or created per grouping, we attach
     * comments (if available) and alerts to the corresponding cases.
     */
    await this.attachCommentAndAlertsToCases(groupedAlertsWithClosedCasesHandled, params);
  }
  groupAlerts({
    params,
    alerts,
    groupingBy
  }) {
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][groupAlerts] Grouping ${alerts.length} alerts`, this.getLogMetadata(params, {
        labels: {
          groupingBy
        },
        tags: ['case-connector:groupAlerts']
      }));
    }
    const uniqueGroupingByFields = Array.from(new Set(groupingBy));
    const groupingMap = new Map();

    /**
     * We are interested in alerts that have a value for any
     * of the groupingBy fields defined by the users. All other
     * alerts will not be attached to any case.
     */
    const [alertsWithAllGroupingFields, noGroupedAlerts] = (0, _lodash.partition)(alerts, alert => uniqueGroupingByFields.every(groupingByField => Boolean((0, _lodash.get)(alert, groupingByField, null))));
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][groupAlerts] Total alerts to be grouped: ${alertsWithAllGroupingFields.length} out of ${alerts.length}`, this.getLogMetadata(params, {
        tags: ['case-connector:groupAlerts']
      }));
    }
    for (const alert of alertsWithAllGroupingFields) {
      const alertWithOnlyTheGroupingFields = (0, _lodash.pick)(alert, uniqueGroupingByFields);
      const groupingKey = (0, _jsonStableStringify.default)(alertWithOnlyTheGroupingFields);
      if (this.logger.isLevelEnabled('debug')) {
        this.logger.debug(`[CasesConnector][CasesConnectorExecutor][groupAlerts] Alert ${alert._id} got grouped into bucket with ID ${groupingKey}`, this.getLogMetadata(params, {
          tags: ['case-connector:groupAlerts', groupingKey]
        }));
      }
      if (groupingMap.has(groupingKey)) {
        var _groupingMap$get;
        (_groupingMap$get = groupingMap.get(groupingKey)) === null || _groupingMap$get === void 0 ? void 0 : _groupingMap$get.alerts.push(alert);
      } else {
        groupingMap.set(groupingKey, {
          alerts: [alert],
          grouping: alertWithOnlyTheGroupingFields
        });
      }
    }
    if (noGroupedAlerts.length > 0) {
      const noGroupedGrouping = this.generateNoGroupAlertGrouping(params.groupingBy);
      groupingMap.set((0, _jsonStableStringify.default)(noGroupedGrouping), {
        alerts: noGroupedAlerts,
        grouping: noGroupedGrouping
      });
    }
    return Array.from(groupingMap.values());
  }
  applyCircuitBreakers(params, groupedAlerts) {
    if (groupedAlerts.length > params.maximumCasesToOpen || groupedAlerts.length > _constants2.MAX_OPEN_CASES) {
      const maxCasesCircuitBreaker = Math.min(params.maximumCasesToOpen, _constants2.MAX_OPEN_CASES);
      this.logger.warn(`[CasesConnector][CasesConnectorExecutor][applyCircuitBreakers] Circuit breaker: Grouping definition would create more than the maximum number of allowed cases ${maxCasesCircuitBreaker}. Falling back to one case.`, this.getLogMetadata(params));
      return this.removeGrouping(groupedAlerts);
    }
    return groupedAlerts;
  }
  removeGrouping(groupedAlerts) {
    const allAlerts = groupedAlerts.map(({
      alerts
    }) => alerts).flat();
    return [{
      alerts: allAlerts,
      grouping: {}
    }];
  }
  generateOracleKeys(params, groupedAlerts) {
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][generateOracleKeys] Generating ${groupedAlerts.length} oracle keys`, this.getLogMetadata(params, {
        tags: ['case-connector:generateOracleKeys']
      }));
    }
    const {
      rule,
      owner
    } = params;
    const oracleMap = new Map();
    for (const {
      grouping,
      alerts,
      comments,
      title
    } of groupedAlerts) {
      const getRecordIdParams = {
        ruleId: rule.id,
        grouping,
        owner,
        spaceId: this.spaceId
      };
      const oracleKey = this.casesOracleService.getRecordId(getRecordIdParams);
      if (this.logger.isLevelEnabled('debug')) {
        this.logger.debug(`[CasesConnector][CasesConnectorExecutor][generateOracleKeys] Oracle key ${oracleKey} generated`, this.getLogMetadata(params, {
          labels: {
            params: getRecordIdParams
          },
          tags: ['case-connector:generateOracleKeys', oracleKey]
        }));
      }
      oracleMap.set(oracleKey, {
        oracleKey,
        grouping,
        alerts,
        comments,
        title
      });
    }
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][generateOracleKeys] Total of oracles keys generated ${oracleMap.size}`, this.getLogMetadata(params, {
        tags: ['case-connector:generateOracleKeys']
      }));
    }
    return oracleMap;
  }
  async upsertOracleRecords(params, groupedAlertsWithOracleKey) {
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertOracleRecords] Upserting ${groupedAlertsWithOracleKey.size} oracle records`, this.getLogMetadata(params, {
        tags: ['case-connector:upsertOracleRecords']
      }));
    }
    const bulkCreateReq = [];
    const oracleRecordMap = new Map();
    const addRecordToMap = oracleRecords => {
      for (const record of oracleRecords) {
        if (groupedAlertsWithOracleKey.has(record.id)) {
          const data = groupedAlertsWithOracleKey.get(record.id);
          oracleRecordMap.set(record.id, {
            ...data,
            oracleRecord: record
          });
        }
      }
    };
    const ids = Array.from(groupedAlertsWithOracleKey.values()).map(({
      oracleKey
    }) => oracleKey);
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertOracleRecords] Getting oracle records with ids ${ids}`, this.getLogMetadata(params, {
        tags: ['case-connector:upsertOracleRecords', ...ids]
      }));
    }
    const bulkGetRes = await this.casesOracleService.bulkGetRecords(ids);
    const [bulkGetValidRecords, bulkGetRecordsErrors] = (0, _utils.partitionRecordsByError)(bulkGetRes);
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertOracleRecords] The total number of valid oracle records is ${bulkGetValidRecords.length} and the total number of errors while getting the records is ${bulkGetRecordsErrors.length}`, this.getLogMetadata(params, {
        labels: {
          total: ids.length,
          success: bulkGetValidRecords.length,
          errors: bulkGetRecordsErrors.length
        },
        tags: ['case-connector:upsertOracleRecords']
      }));
    }
    addRecordToMap(bulkGetValidRecords);
    if (bulkGetRecordsErrors.length === 0) {
      return oracleRecordMap;
    }
    const [nonFoundErrors, restOfErrors] = (0, _utils.partitionByNonFoundErrors)(bulkGetRecordsErrors);
    if (this.logger.isLevelEnabled('debug')) {
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertOracleRecords] The total number of non found oracle records is ${nonFoundErrors.length} and the total number of the rest of errors while getting the records is ${restOfErrors.length}`, this.getLogMetadata(params, {
        labels: {
          nonFoundErrors: nonFoundErrors.length,
          restOfErrors: restOfErrors.length
        },
        tags: ['case-connector:upsertOracleRecords']
      }));
    }
    this.handleAndThrowErrors(restOfErrors);
    if (nonFoundErrors.length === 0) {
      return oracleRecordMap;
    }
    for (const error of nonFoundErrors) {
      if (error.id && groupedAlertsWithOracleKey.has(error.id)) {
        var _record$grouping;
        const record = groupedAlertsWithOracleKey.get(error.id);
        bulkCreateReq.push({
          recordId: error.id,
          payload: {
            rules: [{
              id: params.rule.id
            }],
            grouping: (_record$grouping = record === null || record === void 0 ? void 0 : record.grouping) !== null && _record$grouping !== void 0 ? _record$grouping : {}
          }
        });
      }
    }
    const idsToCreate = bulkCreateReq.map(({
      recordId
    }) => recordId);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertOracleRecords] Creating oracle records with ids ${idsToCreate}`, this.getLogMetadata(params, {
      tags: ['case-connector:upsertOracleRecords', ...idsToCreate]
    }));
    const bulkCreateRes = await this.casesOracleService.bulkCreateRecord(bulkCreateReq);
    const [bulkCreateValidRecords, bulkCreateErrors] = (0, _utils.partitionRecordsByError)(bulkCreateRes);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertOracleRecords] The total number of created oracle records is ${bulkCreateValidRecords.length} and the total number of errors while creating the records is ${bulkCreateErrors.length}`, this.getLogMetadata(params, {
      labels: {
        total: idsToCreate.length,
        success: bulkCreateValidRecords.length,
        errors: bulkCreateErrors.length
      },
      tags: ['case-connector:upsertOracleRecords']
    }));
    this.handleAndThrowErrors(bulkCreateErrors);
    addRecordToMap(bulkCreateValidRecords);
    return oracleRecordMap;
  }
  async handleTimeWindow(params, oracleRecordMap) {
    const {
      timeWindow
    } = params;
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][handleTimeWindow] Handling time window ${timeWindow}`, this.getLogMetadata(params, {
      tags: ['case-connector:handleTimeWindow']
    }));
    const oracleRecordMapWithIncreasedCounters = new Map(oracleRecordMap);
    const recordsToIncreaseCounter = Array.from(oracleRecordMap.values()).filter(({
      oracleRecord
    }) => {
      var _oracleRecord$updated;
      return this.isTimeWindowPassed(params, timeWindow, (_oracleRecord$updated = oracleRecord.updatedAt) !== null && _oracleRecord$updated !== void 0 ? _oracleRecord$updated : oracleRecord.createdAt);
    }).map(({
      oracleRecord
    }) => oracleRecord);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][handleTimeWindow] Total oracle records where the time window has passed and their counter will be increased ${recordsToIncreaseCounter.length}`, this.getLogMetadata(params, {
      tags: ['case-connector:handleTimeWindow', ...recordsToIncreaseCounter.map(({
        id
      }) => id)]
    }));
    const bulkUpdateValidRecords = await this.increaseOracleRecordCounter(params, recordsToIncreaseCounter);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][handleTimeWindow] Total oracle records where their counter got increased ${bulkUpdateValidRecords.length}`, this.getLogMetadata(params, {
      tags: ['case-connector:handleTimeWindow']
    }));
    for (const res of bulkUpdateValidRecords) {
      if (oracleRecordMap.has(res.id)) {
        const data = oracleRecordMap.get(res.id);
        oracleRecordMapWithIncreasedCounters.set(res.id, {
          ...data,
          oracleRecord: res
        });
      }
    }
    return oracleRecordMapWithIncreasedCounters;
  }
  async increaseOracleRecordCounter(params, oracleRecords) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][increaseOracleRecordCounter] Increasing the counters of ${oracleRecords.length} oracle records`, this.getLogMetadata(params, {
      tags: ['case-connector:increaseOracleRecordCounter']
    }));
    if (oracleRecords.length === 0) {
      return [];
    }
    const bulkUpdateReq = oracleRecords.map(record => ({
      recordId: record.id,
      version: record.version,
      /**
       * TODO: Add new cases or any other related info
       */
      payload: {
        counter: record.counter + 1
      }
    }));
    const idsToUpdate = bulkUpdateReq.map(({
      recordId
    }) => recordId);
    const bulkUpdateRes = await this.casesOracleService.bulkUpdateRecord(bulkUpdateReq);
    const [bulkUpdateValidRecords, bulkUpdateErrors] = (0, _utils.partitionRecordsByError)(bulkUpdateRes);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertOracleRecords] The total number of updated oracle records is ${bulkUpdateValidRecords.length} and the total number of errors while updating is ${bulkUpdateErrors.length}`, this.getLogMetadata(params, {
      labels: {
        total: idsToUpdate.length,
        success: bulkUpdateValidRecords.length,
        errors: bulkUpdateErrors.length
      },
      tags: ['case-connector:increaseOracleRecordCounter', ...idsToUpdate]
    }));
    this.handleAndThrowErrors(bulkUpdateErrors);
    return bulkUpdateValidRecords;
  }
  isTimeWindowPassed(params, timeWindow, counterLastUpdatedAt) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][isTimeWindowPassed] Validating the time window ${timeWindow} against the timestamp of the last update of the oracle record ${counterLastUpdatedAt}`, this.getLogMetadata(params, {
      tags: ['case-connector:isTimeWindowPassed']
    }));
    const parsedDate = _datemath.default.parse(`now-${timeWindow}`);

    /**
     * TODO: Should we throw? Should we return true to create a new case?
     */
    if (!parsedDate || !parsedDate.isValid()) {
      this.logger.warn(`[CasesConnector][CasesConnectorExecutor][isTimeWindowPassed] Parsing time window error. Parsing value: "${timeWindow}"`, this.getLogMetadata(params));
      return false;
    }
    const counterLastUpdatedAtAsDate = new Date(counterLastUpdatedAt);

    /**
     * TODO: Should we throw? Should we return true to create a new case?
     */
    if (isNaN(counterLastUpdatedAtAsDate.getTime())) {
      this.logger.warn(`[CasesConnector][CasesConnectorExecutor][isTimeWindowPassed] Timestamp "${counterLastUpdatedAt}" is not a valid date`, this.getLogMetadata(params));
      return false;
    }
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][isTimeWindowPassed] Time window has passed ${counterLastUpdatedAtAsDate < parsedDate.toDate()}`, this.getLogMetadata(params, {
      tags: ['case-connector:isTimeWindowPassed']
    }));
    return counterLastUpdatedAtAsDate < parsedDate.toDate();
  }
  generateCaseIds(params, groupedAlertsWithOracleRecords) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][generateCaseIds] Generating ${groupedAlertsWithOracleRecords.size} case IDs`, this.getLogMetadata(params, {
      tags: ['case-connector:generateCaseIds']
    }));
    const {
      rule,
      owner
    } = params;
    const casesMap = new Map();
    for (const [recordId, entry] of groupedAlertsWithOracleRecords.entries()) {
      const getCaseIdParams = {
        ruleId: rule.id,
        grouping: entry.grouping,
        owner,
        spaceId: this.spaceId,
        counter: entry.oracleRecord.counter
      };
      const caseId = this.casesService.getCaseId(getCaseIdParams);
      this.logger.debug(`[CasesConnector][CasesConnectorExecutor][generateCaseIds] Case ID ${caseId} generated with params ${JSON.stringify(getCaseIdParams)}`, this.getLogMetadata(params, {
        labels: {
          params: getCaseIdParams
        },
        tags: ['case-connector:generateCaseIds', caseId]
      }));
      casesMap.set(caseId, {
        caseId,
        alerts: entry.alerts,
        grouping: entry.grouping,
        comments: entry.comments,
        title: entry.title,
        oracleKey: recordId,
        oracleRecord: entry.oracleRecord
      });
    }
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][generateCaseIds] Total of case IDs generated ${casesMap.size}`, this.getLogMetadata(params, {
      tags: ['case-connector:generateCaseIds']
    }));
    return casesMap;
  }
  async upsertCases(params, groupedAlertsWithCaseId) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertCases] Upserting ${groupedAlertsWithCaseId.size} cases`, this.getLogMetadata(params, {
      tags: ['case-connector:upsertCases']
    }));
    const bulkCreateReq = [];
    const casesMap = new Map();
    const ids = Array.from(groupedAlertsWithCaseId.values()).map(({
      caseId
    }) => caseId);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertCases] Getting cases with ids ${ids}`, this.getLogMetadata(params, {
      tags: ['case-connector:upsertCases', ...ids]
    }));
    const {
      cases,
      errors
    } = await this.casesClient.cases.bulkGet({
      ids
    });
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertCases] The total number of cases is ${cases.length} and the total number of errors while getting the cases is ${errors.length}`, this.getLogMetadata(params, {
      labels: {
        total: ids.length,
        success: cases.length,
        errors: errors.length
      },
      tags: ['case-connector:upsertCases']
    }));
    for (const theCase of cases) {
      if (groupedAlertsWithCaseId.has(theCase.id)) {
        const data = groupedAlertsWithCaseId.get(theCase.id);
        casesMap.set(theCase.id, {
          ...data,
          theCase
        });
      }
    }
    if (errors.length === 0) {
      return casesMap;
    }
    const [nonFoundErrors, restOfErrors] = (0, _utils.partitionByNonFoundErrors)(
    /**
     * The format of error returned from bulkGet is different
     * from what expected. We need to transform to a SavedObjectError
     */
    errors.map(error => {
      var _error$status;
      return {
        ...error,
        statusCode: (_error$status = error.status) !== null && _error$status !== void 0 ? _error$status : 500,
        id: error.caseId
      };
    }));
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertCases] The total number of non found cases is ${nonFoundErrors.length} and the total number of the rest of errors while getting the cases is ${restOfErrors.length}`, this.getLogMetadata(params, {
      labels: {
        nonFoundErrors: nonFoundErrors.length,
        restOfErrors: restOfErrors.length
      },
      tags: ['case-connector:upsertCases']
    }));
    this.handleAndThrowErrors(restOfErrors);
    if (nonFoundErrors.length === 0) {
      return casesMap;
    }
    const {
      customFieldsConfigurationMap,
      templatesConfigurationMap
    } = await this.getCustomFieldsAndTemplatesConfiguration();
    for (const error of nonFoundErrors) {
      if (groupedAlertsWithCaseId.has(error.caseId)) {
        const data = groupedAlertsWithCaseId.get(error.caseId);
        bulkCreateReq.push(this.getCreateCaseRequest(params, data, customFieldsConfigurationMap.get(params.owner), templatesConfigurationMap.get(params.owner)));
      }
    }
    const idsToCreate = bulkCreateReq.map(({
      id
    }) => id);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertCases] Creating cases with ids ${idsToCreate}`, this.getLogMetadata(params, {
      tags: ['case-connector:upsertCases', ...idsToCreate]
    }));

    /**
     * cases.bulkCreate throws an error on errors
     */
    const bulkCreateCasesResponse = await this.casesClient.cases.bulkCreate({
      cases: bulkCreateReq
    });
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][upsertCases] The total number of created cases is ${bulkCreateCasesResponse.cases.length}`, this.getLogMetadata(params, {
      labels: {
        total: bulkCreateReq.length
      },
      tags: ['case-connector:upsertCases']
    }));
    for (const theCase of bulkCreateCasesResponse.cases) {
      if (groupedAlertsWithCaseId.has(theCase.id)) {
        const data = groupedAlertsWithCaseId.get(theCase.id);
        casesMap.set(theCase.id, {
          ...data,
          theCase
        });
      }
    }
    return casesMap;
  }
  getCreateCaseRequest(params, groupingData, customFieldsConfigurations, templatesConfigurations) {
    var _caseFieldsFromTempla, _ref, _caseFieldsFromTempla2, _caseFieldsFromTempla3;
    const {
      grouping,
      caseId,
      oracleRecord,
      title
    } = groupingData;
    const flattenGrouping = (0, _std.getFlattenedObject)(grouping);
    const selectedTemplate = templatesConfigurations === null || templatesConfigurations === void 0 ? void 0 : templatesConfigurations.find(template => template.key === params.templateId);
    const caseFieldsFromTemplate = selectedTemplate ? selectedTemplate.caseFields : null;
    const builtCustomFields = (0, _utils.buildCustomFieldsForRequest)(customFieldsConfigurations, caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.customFields);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][getCreateCaseRequest] Built ${builtCustomFields.length} required custom fields for case with id ${caseId}`, this.getLogMetadata(params, {
      labels: {
        caseId,
        totalCreatedCustomFields: builtCustomFields.length
      },
      tags: ['case-connector:getCreateCaseRequest']
    }));
    return {
      id: caseId,
      description: (_caseFieldsFromTempla = caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.description) !== null && _caseFieldsFromTempla !== void 0 ? _caseFieldsFromTempla : this.getCaseDescription(params, flattenGrouping),
      tags: this.getCaseTags(params, flattenGrouping, caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.tags),
      title: (_ref = title !== null && title !== void 0 ? title : caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.title) !== null && _ref !== void 0 ? _ref : this.getCasesTitle(params, flattenGrouping, oracleRecord.counter),
      connector: (_caseFieldsFromTempla2 = caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.connector) !== null && _caseFieldsFromTempla2 !== void 0 ? _caseFieldsFromTempla2 : {
        id: 'none',
        name: 'none',
        type: _common.ConnectorTypes.none,
        fields: null
      },
      /**
       * TODO: Turn on for Security solution
       */
      settings: (_caseFieldsFromTempla3 = caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.settings) !== null && _caseFieldsFromTempla3 !== void 0 ? _caseFieldsFromTempla3 : {
        syncAlerts: false,
        extractObservables: false
      },
      ...(caseFieldsFromTemplate !== null && caseFieldsFromTemplate !== void 0 && caseFieldsFromTemplate.assignees ? {
        assignees: caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.assignees
      } : {}),
      ...(caseFieldsFromTemplate !== null && caseFieldsFromTemplate !== void 0 && caseFieldsFromTemplate.severity ? {
        severity: caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.severity
      } : {}),
      ...(caseFieldsFromTemplate !== null && caseFieldsFromTemplate !== void 0 && caseFieldsFromTemplate.category ? {
        category: caseFieldsFromTemplate === null || caseFieldsFromTemplate === void 0 ? void 0 : caseFieldsFromTemplate.category
      } : null),
      owner: params.owner,
      customFields: builtCustomFields
    };
  }
  getCasesTitle(params, grouping, oracleCounter) {
    const totalDots = 3;
    const groupingDescription = Object.entries(grouping).map(([_, value]) => {
      return (0, _utils.convertValueToString)(value);
    }).join(' & ');
    const oracleCounterString = oracleCounter > _constants2.INITIAL_ORACLE_RECORD_COUNTER ? ` (${oracleCounter})` : '';
    const staticSuffixStart = groupingDescription.length > 0 ? ` - ${(0, _translations.GROUPED_BY_TITLE)('')}` : '';
    const staticSuffixEnd = `${oracleCounterString} (${_translations.AUTO_CREATED_TITLE})`;
    const ruleName = params.rule.name;
    const ruleNameTrimmed = ruleName.length > _constants.MAX_RULE_NAME_LENGTH ? ruleName.slice(0, _constants.MAX_RULE_NAME_LENGTH - totalDots) : ruleName;
    const ruleNameTrimmedWithDots = ruleName.length > ruleNameTrimmed.length ? `${ruleNameTrimmed}${'.'.repeat(totalDots)}` : ruleNameTrimmed;
    const maxSuffixLength = ruleNameTrimmedWithDots.length < _constants.MAX_RULE_NAME_LENGTH ? _constants.MAX_TITLE_LENGTH - ruleNameTrimmedWithDots.length : _constants.MAX_SUFFIX_LENGTH;
    const maxGroupingDescriptionLength = maxSuffixLength - (staticSuffixStart.length + staticSuffixEnd.length);
    const groupingDescriptionTrimmed = groupingDescription.length > maxGroupingDescriptionLength ? `${groupingDescription.slice(0, maxGroupingDescriptionLength - totalDots)}${'.'.repeat(totalDots)}` : groupingDescription;
    const suffix = `${staticSuffixStart}${groupingDescriptionTrimmed}${staticSuffixEnd}`;
    return `${ruleNameTrimmedWithDots}${suffix}`;
  }
  getCaseDescription(params, grouping) {
    const ruleName = params.rule.ruleUrl ? `['${params.rule.name}'](${params.rule.ruleUrl})` : params.rule.name;
    const description = `${(0, _translations.CASE_CREATED_BY_RULE_DESC)(ruleName)}.`;
    const groupingDescription = Object.entries(grouping).map(([key, value]) => {
      return `\`${key}: ${(0, _utils.convertValueToString)(value)}\``;
    }).join(' and ');
    if (groupingDescription.length > 0) {
      return `${description} ${(0, _translations.GROUPED_BY_DESC)(groupingDescription)}.`;
    }
    return description;
  }
  getCaseTags(params, grouping, templateCaseTags) {
    const ruleTags = Array.isArray(params.rule.tags) ? params.rule.tags : [];
    return ['auto-generated', `rule:${params.rule.id}`, ...this.getGroupingAsTags(grouping), ...ruleTags, ...(templateCaseTags !== null && templateCaseTags !== void 0 ? templateCaseTags : [])].splice(0, _constants.MAX_TAGS_PER_CASE).map(tag => tag.slice(0, _constants.MAX_LENGTH_PER_TAG));
  }
  getGroupingAsTags(grouping) {
    return Object.entries(grouping).map(([key, value]) => [key, `${key}:${(0, _utils.convertValueToString)(value)}`]).flat();
  }
  async handleClosedCases(params, casesMap) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][handleClosedCases] Handling closed cases with reopenClosedCases set to ${params.reopenClosedCases}`, this.getLogMetadata(params, {
      tags: ['case-connector:handleClosedCases']
    }));
    const entriesWithClosedCases = Array.from(casesMap.values()).filter(theCase => theCase.theCase.status === _casesComponents.CaseStatuses.closed);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][handleClosedCases] Closed cases ${entriesWithClosedCases.length}`, this.getLogMetadata(params, {
      tags: ['case-connector:handleClosedCases']
    }));
    if (entriesWithClosedCases.length === 0) {
      return casesMap;
    }
    const res = params.reopenClosedCases ? await this.reopenClosedCases(params, entriesWithClosedCases, casesMap) : await this.createNewCasesOutOfClosedCases(params, entriesWithClosedCases, casesMap);

    /**
     * The initial map contained the closed cases. We need to remove them to
     * avoid attaching alerts to a close case
     */
    return new Map([...res].filter(([_, record]) => record.theCase.status !== _casesComponents.CaseStatuses.closed));
  }
  async reopenClosedCases(params, closedCasesEntries, casesMap) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][reopenClosedCases] Total closed cases to reopen ${closedCasesEntries.length}`, this.getLogMetadata(params, {
      tags: ['case-connector:reopenClosedCases']
    }));
    const casesMapWithClosedCasesOpened = new Map(casesMap);
    const bulkUpdateReq = closedCasesEntries.map(entry => ({
      id: entry.theCase.id,
      version: entry.theCase.version,
      status: _casesComponents.CaseStatuses.open
    }));
    const idsToReopen = bulkUpdateReq.map(({
      id
    }) => id);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][reopenClosedCases] Reopening total ${bulkUpdateReq.length} closed cases with ids ${idsToReopen}`, this.getLogMetadata(params, {
      tags: ['case-connector:reopenClosedCases', ...idsToReopen]
    }));

    /**
     * cases.bulkUpdate throws an error on errors
     */
    const bulkUpdateCasesResponse = await this.casesClient.cases.bulkUpdate({
      cases: bulkUpdateReq
    });
    for (const res of bulkUpdateCasesResponse) {
      if (casesMap.has(res.id)) {
        const data = casesMap.get(res.id);
        casesMapWithClosedCasesOpened.set(res.id, {
          ...data,
          theCase: res
        });
      }
    }
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][reopenClosedCases] The total number of cases that got reopened is ${bulkUpdateCasesResponse.length}`, this.getLogMetadata(params, {
      labels: {
        total: bulkUpdateCasesResponse.length
      },
      tags: ['case-connector:reopenClosedCases']
    }));
    return casesMapWithClosedCasesOpened;
  }
  async createNewCasesOutOfClosedCases(params, closedCasesEntries, casesMap) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][createNewCasesOutOfClosedCases] Creating new cases for closed cases ${closedCasesEntries.length}`, this.getLogMetadata(params, {
      tags: ['case-connector:createNewCasesOutOfClosedCases']
    }));
    const casesMapWithNewCases = new Map(casesMap);
    const casesMapAsArray = Array.from(casesMap.values());
    const findEntryByOracleRecord = oracleId => {
      return casesMapAsArray.find(record => record.oracleRecord.id === oracleId);
    };
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][createNewCasesOutOfClosedCases] Total oracle records where their corresponding case is closed and their counter will be increased ${closedCasesEntries.length}`, this.getLogMetadata(params, {
      tags: ['case-connector:createNewCasesOutOfClosedCases']
    }));
    const bulkUpdateOracleValidRecords = await this.increaseOracleRecordCounter(params, closedCasesEntries.map(entry => entry.oracleRecord));
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][createNewCasesOutOfClosedCases] Total oracle records where their corresponding case is closed and their counter got increased ${bulkUpdateOracleValidRecords.length}`, this.getLogMetadata(params, {
      tags: ['case-connector:createNewCasesOutOfClosedCases', ...closedCasesEntries.map(({
        oracleKey
      }) => oracleKey)]
    }));
    const groupedAlertsWithOracleRecords = new Map();
    for (const record of bulkUpdateOracleValidRecords) {
      const foundRecord = findEntryByOracleRecord(record.id);
      if (foundRecord) {
        groupedAlertsWithOracleRecords.set(record.id, {
          oracleKey: record.id,
          oracleRecord: foundRecord.oracleRecord,
          alerts: foundRecord.alerts,
          grouping: foundRecord.grouping
        });
      }
    }
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][createNewCasesOutOfClosedCases] Generating ${groupedAlertsWithOracleRecords.size} case IDs`, this.getLogMetadata(params, {
      tags: ['case-connector:createNewCasesOutOfClosedCases']
    }));
    const groupedAlertsWithCaseId = this.generateCaseIds(params, groupedAlertsWithOracleRecords);
    const {
      customFieldsConfigurationMap,
      templatesConfigurationMap
    } = await this.getCustomFieldsAndTemplatesConfiguration();
    const bulkCreateReq = Array.from(groupedAlertsWithCaseId.values()).map(record => this.getCreateCaseRequest(params, record, customFieldsConfigurationMap.get(params.owner), templatesConfigurationMap.get(params.owner)));
    const idsToCreate = bulkCreateReq.map(({
      id
    }) => id);
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][createNewCasesOutOfClosedCases] Creating cases with ids ${idsToCreate}`, this.getLogMetadata(params, {
      tags: ['case-connector:createNewCasesOutOfClosedCases', ...idsToCreate]
    }));

    /**
     * cases.bulkCreate throws an error on errors
     */
    const bulkCreateCasesResponse = await this.casesClient.cases.bulkCreate({
      cases: bulkCreateReq
    });
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][createNewCasesOutOfClosedCases] The total number of created cases is ${bulkCreateCasesResponse.cases.length}`, this.getLogMetadata(params, {
      labels: {
        total: bulkCreateCasesResponse.cases.length
      },
      tags: ['case-connector:createNewCasesOutOfClosedCases']
    }));
    for (const theCase of bulkCreateCasesResponse.cases) {
      if (groupedAlertsWithCaseId.has(theCase.id)) {
        const data = groupedAlertsWithCaseId.get(theCase.id);
        casesMapWithNewCases.set(theCase.id, {
          ...data,
          theCase
        });
      }
    }
    return casesMapWithNewCases;
  }
  async attachCommentAndAlertsToCases(groupedAlertsWithCases, params) {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][attachAlertsToCases] Attaching alerts to ${groupedAlertsWithCases.size} cases`, this.getLogMetadata(params, {
      tags: ['case-connector:attachAlertsToCases']
    }));
    const {
      internallyManagedAlerts,
      rule
    } = params;
    const [casesUnderAlertLimit, casesOverAlertLimit] = (0, _lodash.partition)(Array.from(groupedAlertsWithCases.values()), ({
      theCase,
      alerts
    }) => theCase.totalAlerts + alerts.length <= _constants.MAX_ALERTS_PER_CASE);
    if (casesOverAlertLimit.length > 0) {
      const ids = casesOverAlertLimit.map(({
        theCase
      }) => theCase.id);
      const totalAlerts = casesOverAlertLimit.map(({
        alerts
      }) => alerts.length).flat().length;
      this.logger.warn(`Cases with ids "${ids.join(',')}" contain more than ${_constants.MAX_ALERTS_PER_CASE} alerts. The new alerts will not be attached to the cases. Total new alerts: ${totalAlerts}`, this.getLogMetadata(params));
    }
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][attachAlertsToCases] Attaching alerts to ${casesUnderAlertLimit.length} cases that do not have reach the alert limit per case`, this.getLogMetadata(params, {
      tags: ['case-connector:attachAlertsToCases', ...casesUnderAlertLimit.map(({
        caseId
      }) => caseId)]
    }));
    const bulkCreateAlertsRequest = casesUnderAlertLimit.map(({
      theCase,
      alerts,
      comments
    }) => {
      var _comments$map;
      const extraComments = (_comments$map = comments === null || comments === void 0 ? void 0 : comments.map(comment => ({
        type: _common.AttachmentType.user,
        comment,
        owner: theCase.owner
      }))) !== null && _comments$map !== void 0 ? _comments$map : [];
      return {
        caseId: theCase.id,
        attachments: [...extraComments, {
          type: _common.AttachmentType.alert,
          rule: internallyManagedAlerts ? {
            id: null,
            name: null
          } : {
            id: rule.id,
            name: rule.name
          },
          /**
           * Map traverses the array in ascending order.
           * The order is guaranteed to be the same for
           * both calls by the ECMA-262 spec.
           */
          alertId: alerts.map(alert => alert._id),
          index: alerts.map(alert => alert._index),
          owner: theCase.owner
        }]
      };
    });
    await (0, _pMap.default)(bulkCreateAlertsRequest,
    /**
     * attachments.bulkCreate throws an error on errors
     */
    async req => {
      if (this.logger.isLevelEnabled('debug')) {
        this.logger.debug(`[CasesConnector][CasesConnectorExecutor][attachAlertsToCases] Attaching ${req.attachments.length} alerts to case with ID ${req.caseId}`, this.getLogMetadata(params, {
          labels: {
            caseId: req.caseId
          },
          tags: ['case-connector:attachAlertsToCases', req.caseId, ...req.attachments.map(({
            alertId
          }) => alertId)]
        }));
      }
      await this.casesClient.attachments.bulkCreate(req);
    }, {
      concurrency: _constants2.MAX_CONCURRENT_ES_REQUEST
    });
  }
  handleAndThrowErrors(errors) {
    if (errors.length === 0) {
      return;
    }
    const firstError = errors[0];
    const message = `${firstError.error}: ${firstError.message}`;
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][handleAndThrowErrors] Error message "${message}"`, {
      tags: ['case-connector:handleAndThrowErrors'],
      error: {
        code: firstError.statusCode.toString()
      }
    });
    throw new _cases_connector_error.CasesConnectorError(message, firstError.statusCode);
  }
  getLogMetadata(params, {
    tags = [],
    labels = {}
  } = {}) {
    return {
      tags: ['cases-connector', `rule:${params.rule.id}`, ...tags],
      labels
    };
  }
  async getCustomFieldsAndTemplatesConfiguration() {
    this.logger.debug(`[CasesConnector][CasesConnectorExecutor][getCustomFieldsConfiguration] Getting case configurations`, {
      tags: ['case-connector:getCustomFieldsConfiguration']
    });
    const configurations = await this.casesClient.configure.get();
    const customFieldsConfigurationMap = new Map(configurations.map(conf => [conf.owner, conf.customFields]));
    const templatesConfigurationMap = new Map(configurations.map(config => [config.owner, config.templates]));
    return {
      customFieldsConfigurationMap,
      templatesConfigurationMap
    };
  }
}
exports.CasesConnectorExecutor = CasesConnectorExecutor;