"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateUrl = exports.validateIp = exports.validateGenericValue = exports.validateEmail = exports.validateDomain = exports.getValidatorForObservableType = exports.createStringValidationFunction = void 0;
var _ipaddr = _interopRequireDefault(require("ipaddr.js"));
var _emailAddresses = require("email-addresses");
var _lodash = require("lodash");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * DOMAIN REGEX breakdown:
 *
 * (?=.{1,253}$)
 *    Ensures the total length of the domain name is between 1 and 253 characters.
 *
 * ((?!.*--)([a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)\.)+
 *    Matches one or more labels separated by dots. Each label:
 *      - Begins and ends with an alphanumeric character (`[a-zA-Z0-9]`).
 *      - Can contain hyphens (`-`) but not consecutively or at the start/end.
 *      - Has a maximum length of 63 characters.
 *
 * [a-zA-Z]{2,63}
 *    Ensures the top-level domain (TLD) is alphabetic and between 2 and 63 characters long
 *
 * \.?$
 *    Matched domains ending with dot (.) as they are valid domains
 *    @see https://datatracker.ietf.org/doc/html/rfc1034#:~:text=When%20a%20user,ISI.EDU%20domain
 */
const DOMAIN_REGEX = /^(?=.{1,253}$)((?!.*--)([a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)\.)+[a-zA-Z]{2,63}\.?$/;
const GENERIC_REGEX = /^[a-zA-Z0-9._:/\\-]+$/;
const createStringValidationFunction = stringValidator => value => {
  if (!(0, _lodash.isString)(value)) {
    return {
      code: 'ERR_NOT_STRING'
    };
  }
  if (!value.length) {
    return {
      code: 'ERR_EMPTY'
    };
  }
  return stringValidator(value);
};
exports.createStringValidationFunction = createStringValidationFunction;
const validateDomain = exports.validateDomain = createStringValidationFunction(value => {
  return DOMAIN_REGEX.test(value) ? undefined : {
    code: 'ERR_NOT_VALID'
  };
});
const validateGenericValue = exports.validateGenericValue = createStringValidationFunction(value => {
  return GENERIC_REGEX.test(value) ? undefined : {
    code: 'ERR_NOT_VALID'
  };
});
const validateIp = kind => createStringValidationFunction(value => {
  try {
    const parsed = _ipaddr.default.parse(value);
    if (parsed.kind() !== kind) {
      return {
        code: 'ERR_NOT_VALID'
      };
    }
  } catch (error) {
    return {
      code: 'ERR_NOT_VALID'
    };
  }
});
exports.validateIp = validateIp;
const validateUrl = exports.validateUrl = createStringValidationFunction(value => {
  try {
    new URL(value);
  } catch (error) {
    return {
      code: 'ERR_NOT_VALID'
    };
  }
});
const validateEmail = exports.validateEmail = createStringValidationFunction(value => {
  if ((0, _emailAddresses.parseAddressList)(value) === null) {
    return {
      code: 'ERR_NOT_EMAIL'
    };
  }
});
const getValidatorForObservableType = observableTypeKey => {
  switch (observableTypeKey) {
    case _constants.OBSERVABLE_TYPE_URL.key:
      {
        return validateUrl;
      }
    case _constants.OBSERVABLE_TYPE_DOMAIN.key:
      {
        return validateDomain;
      }
    case _constants.OBSERVABLE_TYPE_EMAIL.key:
      {
        return validateEmail;
      }
    case _constants.OBSERVABLE_TYPE_IPV4.key:
      {
        return validateIp('ipv4');
      }
    case _constants.OBSERVABLE_TYPE_IPV6.key:
      {
        return validateIp('ipv6');
      }
    default:
      {
        return validateGenericValue;
      }
  }
};
exports.getValidatorForObservableType = getValidatorForObservableType;