"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAllRefValues = void 0;
exports.getAuthDetails = getAuthDetails;
exports.reduceSpecComponents = reduceSpecComponents;
var _cel_input_attributes = require("../../common/api/model/cel_input_attributes.gen");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/no-explicit-any */

/**
 * Returns any $ref from the specified schema object.
 */
const getAllRefValues = schemaObj => {
  let allRefs = new Set();
  if (schemaObj === null || typeof schemaObj !== 'object') {
    return allRefs;
  }
  if (Array.isArray(schemaObj)) {
    for (const elem of schemaObj) {
      if (typeof elem === 'object') {
        const subRefs = getAllRefValues(elem);
        if (subRefs.size > 0) {
          allRefs = new Set([...allRefs, ...subRefs]);
        }
      }
    }
    return allRefs;
  }
  for (const [key, value] of Object.entries(schemaObj)) {
    if (key === '$ref' && typeof value === 'string') {
      allRefs.add(value);
    } else if (typeof value === 'object' && value !== null) {
      const subRefs = getAllRefValues(value);
      if (subRefs.size > 0) {
        allRefs = new Set([...allRefs, ...subRefs]);
      }
    }
  }
  return allRefs;
};

/**
 * Retrieves the OAS spec components down to only those utilized by the specified path.
 */
exports.getAllRefValues = getAllRefValues;
function reduceSpecComponents(oas, path) {
  var _oas$operation;
  const responses = oas === null || oas === void 0 ? void 0 : (_oas$operation = oas.operation(path, 'get')) === null || _oas$operation === void 0 ? void 0 : _oas$operation.schema.responses;
  const usedSchemas = getAllRefValues(responses);
  if (oas !== null && oas !== void 0 && oas.api) {
    // iterate the schemas and remove those not used
    const reduced = JSON.parse(JSON.stringify(oas === null || oas === void 0 ? void 0 : oas.api.components));
    if (reduced) {
      for (const [componentType, items] of Object.entries(reduced)) {
        for (const component of Object.keys(items)) {
          if (!usedSchemas.has(`#/components/${componentType}/${component}`)) {
            var _reduced;
            (_reduced = reduced[componentType]) === null || _reduced === void 0 ? true : delete _reduced[component];
          }
        }
        if (Object.keys(items).length < 1) {
          delete reduced[componentType];
        }
      }
    }
    return reduced;
  }
}

/**
 * Maps the cel authType to the corresponding auth details from the OAS schema.
 */
function getAuthDetails(authType, specAuthDetails) {
  const auth = authType.toLowerCase();
  if (auth === _cel_input_attributes.CelAuthTypeEnum.basic) {
    return (specAuthDetails === null || specAuthDetails === void 0 ? void 0 : specAuthDetails.Basic[0]) || undefined;
  } else if (auth === _cel_input_attributes.CelAuthTypeEnum.oauth2) {
    return (specAuthDetails === null || specAuthDetails === void 0 ? void 0 : specAuthDetails.OAuth2[0]) || undefined;
  } else if (auth === _cel_input_attributes.CelAuthTypeEnum.header) {
    var _specAuthDetails$Head, _specAuthDetails$Bear, _specAuthDetails$apiK;
    return (specAuthDetails === null || specAuthDetails === void 0 ? void 0 : (_specAuthDetails$Head = specAuthDetails.Header) === null || _specAuthDetails$Head === void 0 ? void 0 : _specAuthDetails$Head[0]) || (specAuthDetails === null || specAuthDetails === void 0 ? void 0 : (_specAuthDetails$Bear = specAuthDetails.Bearer) === null || _specAuthDetails$Bear === void 0 ? void 0 : _specAuthDetails$Bear[0]) || (specAuthDetails === null || specAuthDetails === void 0 ? void 0 : (_specAuthDetails$apiK = specAuthDetails.apiKey) === null || _specAuthDetails$apiK === void 0 ? void 0 : _specAuthDetails$apiK[0]) || undefined;
  } else if (auth === _cel_input_attributes.CelAuthTypeEnum.digest) {
    return (specAuthDetails === null || specAuthDetails === void 0 ? void 0 : specAuthDetails.http[0]) || undefined;
  } else {
    // should never get here
    throw new Error('unsupported auth method');
  }
}