"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLatencyPeriods = getLatencyPeriods;
exports.getLatencyTimeseries = getLatencyTimeseries;
var _server = require("@kbn/observability-plugin/server");
var _apm = require("../../../../common/es_fields/apm");
var _environment_query = require("../../../../common/utils/environment_query");
var _get_offset_in_ms = require("../../../../common/utils/get_offset_in_ms");
var _offset_previous_period_coordinate = require("../../../../common/utils/offset_previous_period_coordinate");
var _latency_aggregation_type = require("../../../lib/helpers/latency_aggregation_type");
var _transactions = require("../../../lib/helpers/transactions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function searchLatency({
  environment,
  kuery,
  filters,
  serviceName,
  transactionType,
  transactionName,
  apmEventClient,
  latencyAggregationType,
  start,
  end,
  offset,
  serverlessId,
  documentType,
  rollupInterval,
  bucketSizeInSeconds,
  useDurationSummary
}) {
  const {
    startWithOffset,
    endWithOffset
  } = (0, _get_offset_in_ms.getOffsetInMs)({
    start,
    end,
    offset
  });
  const transactionDurationField = (0, _transactions.getDurationFieldForTransactions)(documentType, useDurationSummary);
  const params = {
    apm: {
      sources: [{
        documentType,
        rollupInterval
      }]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            [_apm.SERVICE_NAME]: serviceName
          }
        }, ...(0, _server.rangeQuery)(startWithOffset, endWithOffset), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery), ...(0, _server.termQuery)(_apm.TRANSACTION_NAME, transactionName), ...(0, _server.termQuery)(_apm.TRANSACTION_TYPE, transactionType), ...(0, _server.termQuery)(_apm.FAAS_ID, serverlessId), ...((filters === null || filters === void 0 ? void 0 : filters.filter) || [])],
        must_not: (filters === null || filters === void 0 ? void 0 : filters.must_not) || []
      }
    },
    aggs: {
      latencyTimeseries: {
        date_histogram: {
          field: '@timestamp',
          fixed_interval: `${bucketSizeInSeconds}s`,
          min_doc_count: 0,
          extended_bounds: {
            min: startWithOffset,
            max: endWithOffset
          }
        },
        aggs: (0, _latency_aggregation_type.getLatencyAggregation)(latencyAggregationType, transactionDurationField)
      },
      overall_avg_duration: {
        avg: {
          field: transactionDurationField
        }
      }
    }
  };
  return apmEventClient.search('get_latency_charts', params);
}
async function getLatencyTimeseries({
  environment,
  kuery,
  filters,
  serviceName,
  transactionType,
  transactionName,
  apmEventClient,
  latencyAggregationType,
  start,
  end,
  offset,
  serverlessId,
  documentType,
  rollupInterval,
  bucketSizeInSeconds,
  useDurationSummary
}) {
  const response = await searchLatency({
    environment,
    kuery,
    filters,
    serviceName,
    transactionType,
    transactionName,
    apmEventClient,
    latencyAggregationType,
    start,
    end,
    offset,
    serverlessId,
    documentType,
    rollupInterval,
    bucketSizeInSeconds,
    useDurationSummary
  });
  if (!response.aggregations) {
    return {
      latencyTimeseries: [],
      overallAvgDuration: null
    };
  }
  return {
    overallAvgDuration: response.aggregations.overall_avg_duration.value || null,
    latencyTimeseries: response.aggregations.latencyTimeseries.buckets.map(bucket => {
      return {
        x: bucket.key,
        y: (0, _latency_aggregation_type.getLatencyValue)({
          latencyAggregationType,
          aggregation: bucket.latency
        })
      };
    })
  };
}
async function getLatencyPeriods({
  serviceName,
  transactionType,
  transactionName,
  apmEventClient,
  latencyAggregationType,
  kuery,
  filters,
  environment,
  start,
  end,
  offset,
  documentType,
  rollupInterval,
  bucketSizeInSeconds,
  useDurationSummary
}) {
  const options = {
    serviceName,
    transactionType,
    transactionName,
    apmEventClient,
    kuery,
    filters,
    environment,
    documentType,
    rollupInterval,
    bucketSizeInSeconds,
    useDurationSummary
  };
  const currentPeriodPromise = getLatencyTimeseries({
    ...options,
    start,
    end,
    latencyAggregationType
  });
  const previousPeriodPromise = offset ? getLatencyTimeseries({
    ...options,
    start,
    end,
    offset,
    latencyAggregationType
  }) : {
    latencyTimeseries: [],
    overallAvgDuration: null
  };
  const [currentPeriod, previousPeriod] = await Promise.all([currentPeriodPromise, previousPeriodPromise]);
  return {
    currentPeriod,
    previousPeriod: {
      ...previousPeriod,
      latencyTimeseries: (0, _offset_previous_period_coordinate.offsetPreviousPeriodCoordinates)({
        currentPeriodTimeseries: currentPeriod.latencyTimeseries,
        previousPeriodTimeseries: previousPeriod.latencyTimeseries
      })
    }
  };
}