"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TIME = exports.RATE = void 0;
exports.fetchAndTransformGcMetrics = fetchAndTransformGcMetrics;
var _lodash = require("lodash");
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _is_finite_number = require("../../../../../../common/utils/is_finite_number");
var _metrics = require("../../../../../lib/helpers/metrics");
var _apm = require("../../../../../../common/es_fields/apm");
var _get_bucket_size = require("../../../../../../common/utils/get_bucket_size");
var _agent_name = require("../../../../../../common/agent_name");
var _environment_query = require("../../../../../../common/utils/environment_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RATE = exports.RATE = 'rate';
const TIME = exports.TIME = 'time';
async function fetchAndTransformGcMetrics({
  environment,
  kuery,
  config,
  apmEventClient,
  serviceName,
  serviceNodeName,
  chartBase,
  rateOrTime,
  operationName,
  start,
  end,
  isOpenTelemetry
}) {
  const {
    bucketSize
  } = (0, _get_bucket_size.getBucketSize)({
    start,
    end
  });
  const groupByField = isOpenTelemetry ? _apm.LABEL_GC : _apm.LABEL_NAME;
  const targetField = isOpenTelemetry ? _apm.METRIC_OTEL_JVM_GC_DURATION : rateOrTime === RATE ? _apm.METRIC_JAVA_GC_COUNT : _apm.METRIC_JAVA_GC_TIME;
  const fieldAggregation = isOpenTelemetry ? rateOrTime === RATE ? {
    value_count: {
      field: targetField
    }
  } : {
    sum: {
      field: targetField
    }
  } : {
    max: {
      field: targetField
    }
  };

  // GC rate and time are reported by the agents as monotonically
  // increasing counters, which means that we have to calculate
  // the delta in an es query. In the future agent might start
  // reporting deltas.
  const params = {
    apm: {
      events: [_common.ProcessorEvent.metric]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            [_apm.SERVICE_NAME]: serviceName
          }
        }, ...(0, _environment_query.serviceNodeNameQuery)(serviceNodeName), ...(0, _server.rangeQuery)(start, end), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery), {
          exists: {
            field: targetField
          }
        }, {
          terms: {
            [_apm.AGENT_NAME]: _agent_name.JAVA_AGENT_NAMES
          }
        }]
      }
    },
    aggs: {
      per_pool: {
        terms: {
          field: `${groupByField}`
        },
        aggs: {
          timeseries: {
            date_histogram: (0, _metrics.getMetricsDateHistogramParams)({
              start,
              end,
              metricsInterval: config.metricsInterval
            }),
            aggs: {
              // get the max value
              max: fieldAggregation,
              // get the derivative, which is the delta y
              derivative: {
                derivative: {
                  buckets_path: 'max'
                }
              },
              // if a gc counter is reset, the delta will be >0 and
              // needs to be excluded
              value: {
                bucket_script: {
                  buckets_path: {
                    value: 'derivative'
                  },
                  script: 'params.value > 0.0 ? params.value : 0.0'
                }
              }
            }
          }
        }
      }
    }
  };
  const response = await apmEventClient.search(operationName, params);
  const {
    aggregations
  } = response;
  if (!aggregations) {
    return {
      ...chartBase,
      series: []
    };
  }
  const series = aggregations.per_pool.buckets.map((poolBucket, i) => {
    const label = poolBucket.key;
    const timeseriesData = poolBucket.timeseries;
    const data = timeseriesData.buckets.map(bucket => {
      var _bucket$value;
      // derivative/value will be undefined for the first hit and if the `max` value is null
      const bucketValue = (_bucket$value = bucket.value) === null || _bucket$value === void 0 ? void 0 : _bucket$value.value;
      const unconvertedY = (0, _is_finite_number.isFiniteNumber)(bucketValue) ? (0, _lodash.round)(bucketValue * (60 / bucketSize), 1) : null;

      // convert to milliseconds if we're calculating time, but not for rate
      const y = unconvertedY !== null && rateOrTime === TIME ? unconvertedY * 1000 : unconvertedY;
      return {
        y,
        x: bucket.key
      };
    });
    const values = data.map(coordinate => coordinate.y).filter(y => y !== null);
    const overallValue = (0, _lodash.sum)(values) / values.length;
    return {
      title: label,
      key: label,
      type: chartBase.type,
      overallValue,
      data
    };
  });
  return {
    ...chartBase,
    series
  };
}