"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.groupResourceNodes = groupResourceNodes;
var _i18n = require("@kbn/i18n");
var _apm = require("../es_fields/apm");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MINIMUM_GROUP_SIZE = 4;
const isEdge = el => Boolean(el.data.source && el.data.target);
const isNode = el => !isEdge(el);
const isElligibleGroupNode = el => {
  if (isNode(el) && _apm.SPAN_TYPE in el.data) {
    return (0, _utils.isSpanGroupingSupported)(el.data[_apm.SPAN_TYPE], el.data[_apm.SPAN_SUBTYPE]);
  }
  return false;
};
function groupConnections({
  edgesMap,
  groupableNodeIds
}) {
  const sourcesByTarget = new Map();
  for (const {
    data
  } of edgesMap.values()) {
    const {
      source,
      target
    } = data;
    if (groupableNodeIds.has(target)) {
      var _sourcesByTarget$get;
      const sources = (_sourcesByTarget$get = sourcesByTarget.get(target)) !== null && _sourcesByTarget$get !== void 0 ? _sourcesByTarget$get : [];
      sources.push(source);
      sourcesByTarget.set(target, sources);
    }
  }
  const groups = new Map();
  for (const [target, sources] of sourcesByTarget) {
    var _groups$get;
    const groupId = `resourceGroup{${[...sources].sort().join(';')}}`;
    const group = (_groups$get = groups.get(groupId)) !== null && _groups$get !== void 0 ? _groups$get : {
      id: groupId,
      sources,
      targets: []
    };
    group.targets.push(target);
    groups.set(groupId, group);
  }
  return Array.from(groups.values()).filter(({
    targets
  }) => targets.length >= MINIMUM_GROUP_SIZE);
}
function getUngroupedNodesAndEdges({
  nodesMap,
  edgesMap,
  groupedConnections
}) {
  const ungroupedEdges = new Map(edgesMap);
  const ungroupedNodes = new Map(nodesMap);
  for (const {
    sources,
    targets
  } of groupedConnections) {
    targets.forEach(target => {
      ungroupedNodes.delete(target);
      sources.forEach(source => {
        ungroupedEdges.delete((0, _utils.getEdgeId)(source, target));
      });
    });
  }
  return {
    ungroupedNodes,
    ungroupedEdges
  };
}
function groupNodes({
  nodesMap,
  groupedConnections
}) {
  return groupedConnections.map(({
    id,
    targets
  }) => ({
    data: {
      id,
      [_apm.SPAN_TYPE]: 'external',
      label: _i18n.i18n.translate('xpack.apm.serviceMap.resourceCountLabel', {
        defaultMessage: '{count} resources',
        values: {
          count: targets.length
        }
      }),
      groupedConnections: targets.map(target => {
        const targetElement = nodesMap.get(target);
        return targetElement ? {
          ...targetElement.data,
          label: targetElement.data.label || targetElement.data.id
        } : undefined;
      }).filter(target => !!target)
    }
  }));
}
function groupEdges({
  groupedConnections
}) {
  return groupedConnections.flatMap(({
    id,
    sources
  }) => sources.map(source => ({
    data: {
      id: `${source}~>${id}`,
      source,
      target: id
    }
  })));
}
function groupResourceNodes({
  elements
}) {
  const nodesMap = new Map(elements.filter(isNode).map(node => [node.data.id, node]));
  const edgesMap = new Map(elements.filter(isEdge).map(edge => [(0, _utils.getEdgeId)(edge.data.source, edge.data.target), edge]));
  const groupableNodeIds = new Set(elements.filter(isElligibleGroupNode).map(({
    data: {
      id
    }
  }) => id));
  const groupedConnections = groupConnections({
    edgesMap,
    groupableNodeIds
  });
  const {
    ungroupedEdges,
    ungroupedNodes
  } = getUngroupedNodesAndEdges({
    nodesMap,
    edgesMap,
    groupedConnections
  });
  const groupedNodes = groupNodes({
    nodesMap,
    groupedConnections
  });
  const groupedEdges = groupEdges({
    groupedConnections
  });
  return {
    elements: [...ungroupedNodes.values(), ...groupedNodes, ...ungroupedEdges.values(), ...groupedEdges],
    nodesCount: ungroupedNodes.size
  };
}