"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRuleInMemory = updateRuleInMemory;
var _lib = require("../../../lib");
var _ = require("..");
var _lib2 = require("../../lib");
var _transforms = require("../../../application/rule/transforms");
var _mapped_params_utils = require("../mapped_params_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function updateRuleInMemory(context, {
  rule,
  updateAttributesFn,
  paramsModifier,
  apiKeysMap,
  rules,
  skipped,
  errors,
  username,
  shouldInvalidateApiKeys,
  shouldIncrementRevision = () => true
}) {
  context.logger.info(`Updating rule in memory for rule: ${rule.id}`);
  if (rule.attributes.apiKey) {
    apiKeysMap.set(rule.id, {
      oldApiKey: rule.attributes.apiKey,
      oldApiKeyCreatedByUser: rule.attributes.apiKeyCreatedByUser
    });
  }
  const ruleType = context.ruleTypeRegistry.get(rule.attributes.alertTypeId);
  const ruleActions = (0, _.injectReferencesIntoActions)(rule.id, rule.attributes.actions || [], rule.references || []);
  context.logger.info(`ruleActions ${JSON.stringify(ruleActions)}`);
  const ruleArtifacts = (0, _.injectReferencesIntoArtifacts)(rule.id, rule.attributes.artifacts, rule.references);
  const ruleDomain = (0, _transforms.transformRuleAttributesToRuleDomain)(rule.attributes, {
    id: rule.id,
    logger: context.logger,
    ruleType,
    references: rule.references
  }, context.isSystemAction);
  context.logger.info(`ruleDomain ${JSON.stringify(ruleDomain)}`);
  const {
    rule: updatedRule,
    ruleActions: updatedRuleActions,
    hasUpdateApiKeyOperation,
    isAttributesUpdateSkipped
  } = await updateAttributesFn({
    domainRule: ruleDomain,
    ruleActions,
    ruleType
  });
  context.logger.info(`updatedRule ${JSON.stringify(updatedRule)}`);
  const {
    modifiedParams: ruleParams,
    isParamsUpdateSkipped
  } = paramsModifier ?
  // TODO (http-versioning): Remove the cast when all rule types are fixed
  await paramsModifier(updatedRule) : {
    modifiedParams: updatedRule.params,
    isParamsUpdateSkipped: true
  };

  // Increment revision if params ended up being modified AND it wasn't already incremented as part of attribute update
  if (shouldIncrementRevision(ruleParams) && !isParamsUpdateSkipped && rule.attributes.revision === updatedRule.revision) {
    updatedRule.revision += 1;
  }

  // If neither attributes nor parameters were updated, mark
  // the rule as skipped and continue to the next rule.
  if (isAttributesUpdateSkipped && isParamsUpdateSkipped) {
    skipped.push({
      id: rule.id,
      name: rule.attributes.name,
      skip_reason: 'RULE_NOT_MODIFIED'
    });
    return;
  }

  // validate rule params
  const validatedAlertTypeParams = (0, _lib.validateRuleTypeParams)(ruleParams, ruleType.validate.params);
  const validatedMutatedAlertTypeParams = (0, _lib.validateMutatedRuleTypeParams)(validatedAlertTypeParams, rule.attributes.params, ruleType.validate.params);
  const {
    references,
    params: updatedParams,
    actions: actionsWithRefs,
    artifacts: artifactsWithRefs
  } = await (0, _lib2.extractReferences)(context, ruleType, updatedRuleActions, validatedMutatedAlertTypeParams, ruleArtifacts !== null && ruleArtifacts !== void 0 ? ruleArtifacts : {});
  const ruleAttributes = (0, _transforms.transformRuleDomainToRuleAttributes)({
    actionsWithRefs,
    rule: updatedRule,
    params: {
      legacyId: rule.attributes.legacyId,
      paramsWithRefs: updatedParams
    },
    artifactsWithRefs
  });
  context.logger.info(`ruleAttributes ${JSON.stringify(ruleAttributes)}`);
  let apiKeyAttributes;
  if (shouldInvalidateApiKeys) {
    const {
      apiKeyAttributes: preparedApiKeyAttributes
    } = await prepareApiKeys(context, rule, ruleType, apiKeysMap, ruleAttributes, hasUpdateApiKeyOperation, username);
    apiKeyAttributes = preparedApiKeyAttributes;
  }
  const {
    updatedAttributes
  } = updateAttributes({
    context,
    attributes: ruleAttributes,
    apiKeyAttributes,
    updatedParams,
    rawAlertActions: ruleAttributes.actions,
    username
  });
  context.logger.info(`updatedAttributes ${JSON.stringify(updatedAttributes)}`);
  rules.push({
    ...rule,
    references,
    attributes: updatedAttributes
  });
}
async function prepareApiKeys(context, rule, ruleType, apiKeysMap, attributes, hasUpdateApiKeyOperation, username) {
  const apiKeyAttributes = await (0, _lib2.createNewAPIKeySet)(context, {
    id: ruleType.id,
    ruleName: attributes.name,
    username,
    shouldUpdateApiKey: attributes.enabled || hasUpdateApiKeyOperation,
    errorMessage: 'Error updating rule: could not create API key'
  });

  // collect generated API keys
  if (apiKeyAttributes.apiKey) {
    apiKeysMap.set(rule.id, {
      ...apiKeysMap.get(rule.id),
      newApiKey: apiKeyAttributes.apiKey,
      newApiKeyCreatedByUser: apiKeyAttributes.apiKeyCreatedByUser
    });
  }
  return {
    apiKeyAttributes
  };
}
function updateAttributes({
  context,
  attributes,
  apiKeyAttributes,
  updatedParams,
  rawAlertActions,
  username
}) {
  var _attributes$notifyWhe, _attributes$throttle;
  // get notifyWhen
  const notifyWhen = (0, _lib.getRuleNotifyWhenType)((_attributes$notifyWhe = attributes.notifyWhen) !== null && _attributes$notifyWhe !== void 0 ? _attributes$notifyWhe : null, (_attributes$throttle = attributes.throttle) !== null && _attributes$throttle !== void 0 ? _attributes$throttle : null);

  // TODO (http-versioning) Remove casts when updateMeta has been converted
  const castedAttributes = attributes;
  const updatedAttributes = (0, _lib2.updateMeta)(context, {
    ...castedAttributes,
    ...(apiKeyAttributes ? {
      ...apiKeyAttributes
    } : {}),
    params: updatedParams,
    actions: rawAlertActions,
    notifyWhen,
    updatedBy: username,
    updatedAt: new Date().toISOString()
  });

  // add mapped_params
  const mappedParams = (0, _mapped_params_utils.getMappedParams)(updatedParams);
  if (Object.keys(mappedParams).length) {
    updatedAttributes.mapped_params = mappedParams;
  }
  return {
    updatedAttributes
  };
}