"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkEditRules = bulkEditRules;
var _pMap = _interopRequireDefault(require("p-map"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _esQuery = require("@kbn/es-query");
var _audit_events = require("../audit_events");
var _authorization = require("../../../authorization");
var _build_kuery_node_filter = require("../build_kuery_node_filter");
var _lib = require("../../../lib");
var _constants = require("../constants");
var _rule = require("../../../data/rule");
var _retry_if_bulk_edit_conflicts = require("./retry_if_bulk_edit_conflicts");
var _bulk_mark_api_keys_for_invalidation = require("../../../invalidate_pending_api_keys/bulk_mark_api_keys_for_invalidation");
var _schemas = require("../../../application/rule/schemas");
var _transforms = require("../../../application/rule/transforms");
var _bulk_edit_rules_occ = require("./bulk_edit_rules_occ");
var _construct_ignore_internal_rule_type_filters = require("../construct_ignore_internal_rule_type_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function bulkEditRules(context, options) {
  var _options$ignoreIntern;
  const queryFilter = options.filter;
  const ids = options.ids;
  const ignoreInternalRuleTypes = (_options$ignoreIntern = options.ignoreInternalRuleTypes) !== null && _options$ignoreIntern !== void 0 ? _options$ignoreIntern : true;
  const actionsClient = await context.getActionsClient();
  if (ids && queryFilter) {
    throw _boom.default.badRequest("Both 'filter' and 'ids' are supplied. Define either 'ids' or 'filter' properties in method arguments");
  }
  const qNodeQueryFilter = (0, _build_kuery_node_filter.buildKueryNodeFilter)(queryFilter);
  const qNodeFilter = ids ? (0, _lib.convertRuleIdsToKueryNode)(ids) : qNodeQueryFilter;
  const internalRuleTypeFilter = (0, _construct_ignore_internal_rule_type_filters.constructIgnoreInternalRuleTypesFilter)({
    ruleTypes: context.ruleTypeRegistry.list()
  });
  let authorizationTuple;
  try {
    authorizationTuple = await context.authorization.getFindAuthorizationFilter({
      authorizationEntity: _authorization.AlertingAuthorizationEntity.Rule,
      filterOpts: _constants.alertingAuthorizationFilterOpts
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: options.auditAction,
      error
    }));
    throw error;
  }
  const {
    filter: authorizationFilter
  } = authorizationTuple;
  const qNodeFilterWithAuth = authorizationFilter && qNodeFilter ? _esQuery.nodeBuilder.and([qNodeFilter, authorizationFilter]) : qNodeFilter;
  const finalFilter = ignoreInternalRuleTypes ? (0, _construct_ignore_internal_rule_type_filters.combineFiltersWithInternalRuleTypeFilter)({
    filter: qNodeFilterWithAuth,
    internalRuleTypeFilter
  }) : qNodeFilterWithAuth;
  const {
    aggregations,
    total
  } = await (0, _rule.findRulesSo)({
    savedObjectsClient: context.unsecuredSavedObjectsClient,
    savedObjectsFindOptions: {
      filter: finalFilter,
      page: 1,
      perPage: 0,
      aggs: {
        alertTypeId: {
          multi_terms: {
            terms: [{
              field: 'alert.attributes.alertTypeId'
            }, {
              field: 'alert.attributes.consumer'
            }]
          }
        }
      }
    }
  });
  if (total > _constants.MAX_RULES_NUMBER_FOR_BULK_OPERATION) {
    throw _boom.default.badRequest(`More than ${_constants.MAX_RULES_NUMBER_FOR_BULK_OPERATION} rules matched for bulk edit`);
  }
  const buckets = aggregations === null || aggregations === void 0 ? void 0 : aggregations.alertTypeId.buckets;
  if (buckets === undefined) {
    throw Error('No rules found for bulk edit');
  }
  await (0, _pMap.default)(buckets, async ({
    key: [ruleType, consumer]
  }) => {
    context.ruleTypeRegistry.ensureRuleTypeEnabled(ruleType);
    try {
      await context.authorization.ensureAuthorized({
        ruleTypeId: ruleType,
        consumer,
        operation: options.requiredAuthOperation,
        entity: _authorization.AlertingAuthorizationEntity.Rule
      });
    } catch (error) {
      var _context$auditLogger2;
      (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
        action: options.auditAction,
        error
      }));
      throw error;
    }
  }, {
    concurrency: _constants.RULE_TYPE_CHECKS_CONCURRENCY
  });
  const {
    apiKeysToInvalidate,
    results,
    errors,
    skipped
  } = await (0, _retry_if_bulk_edit_conflicts.retryIfBulkEditConflicts)(context.logger, options.name, filterKueryNode => (0, _bulk_edit_rules_occ.bulkEditRulesOcc)(context, {
    filter: filterKueryNode,
    shouldValidateSchedule: options.shouldValidateSchedule,
    shouldInvalidateApiKeys: options.shouldInvalidateApiKeys,
    updateFn: options.updateFn,
    paramsModifier: options.paramsModifier,
    shouldIncrementRevision: options.shouldIncrementRevision
  }), finalFilter);
  if (apiKeysToInvalidate.length > 0) {
    await (0, _bulk_mark_api_keys_for_invalidation.bulkMarkApiKeysForInvalidation)({
      apiKeys: apiKeysToInvalidate
    }, context.logger, context.unsecuredSavedObjectsClient);
  }
  const updatedRules = results.map(({
    id,
    attributes,
    references
  }) => {
    // TODO (http-versioning): alertTypeId should never be null, but we need to
    // fix the type cast from SavedObjectsBulkUpdateObject to SavedObjectsBulkUpdateObject
    // when we are doing the bulk create and this should fix itself
    const ruleType = context.ruleTypeRegistry.get(attributes.alertTypeId);
    const ruleDomain = (0, _transforms.transformRuleAttributesToRuleDomain)(attributes, {
      id,
      logger: context.logger,
      ruleType,
      references,
      omitGeneratedValues: false
    }, connectorId => actionsClient.isSystemAction(connectorId));
    try {
      _schemas.ruleDomainSchema.validate(ruleDomain);
    } catch (e) {
      context.logger.warn(`Error validating bulk edited rule domain object for id: ${id}, ${e}`);
    }
    return ruleDomain;
  });

  // TODO (http-versioning): This should be of type Rule, change this when all rule types are fixed
  const publicRules = updatedRules.map(rule => {
    return (0, _transforms.transformRuleDomainToRule)(rule);
  });
  return {
    rules: publicRules,
    skipped,
    errors,
    total
  };
}