"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processGapsBatch = void 0;
var _interval_utils = require("../../../../lib/rule_gaps/gap/interval_utils");
var _schedule = require("../../../backfill/methods/schedule");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const processGapsBatch = async (context, {
  rule,
  range,
  gapsBatch,
  maxGapsCountToProcess
}) => {
  const {
    start,
    end
  } = range;
  let processedGapsCount = 0;
  let gapsClampedIntervals = gapsBatch.map(gap => ({
    gap,
    clampedIntervals: (0, _interval_utils.clampIntervals)(gap.unfilledIntervals, {
      gte: new Date(start),
      lte: new Date(end)
    })
  })).filter(({
    clampedIntervals
  }) => clampedIntervals.length > 0);
  if (maxGapsCountToProcess && maxGapsCountToProcess < gapsClampedIntervals.length) {
    gapsClampedIntervals = gapsClampedIntervals.slice(0, Math.min(maxGapsCountToProcess, gapsClampedIntervals.length));
  }
  processedGapsCount += gapsClampedIntervals.length;
  const gapsInBackfillScheduling = gapsClampedIntervals.map(({
    gap
  }) => gap);
  const gapRanges = gapsClampedIntervals.flatMap(({
    clampedIntervals
  }) => clampedIntervals.map(({
    gte,
    lte
  }) => ({
    start: gte.toISOString(),
    end: lte.toISOString()
  })));

  // Rules might have gaps within the range that don't yield any schedulingPayload
  // This can happen when they have gaps that are in an "in progress" state.
  // They are returned still returned by the function that is querying gaps
  if (gapRanges.length === 0) {
    return {
      processedGapsCount: 0
    };
  }
  const schedulingPayload = {
    ruleId: rule.id,
    ranges: gapRanges
  };
  const results = await (0, _schedule.scheduleBackfill)(context, [schedulingPayload], gapsInBackfillScheduling);
  if (results.length !== 1) {
    throw new Error(`Unexpected scheduling result count ${results.length}`);
  } else if ('error' in results[0]) {
    const backfillError = results[0].error;
    throw new Error(backfillError.message);
  } else {
    (0, _utils.logProcessedAsAuditEvent)(context, rule);
  }
  return {
    processedGapsCount
  };
};
exports.processGapsBatch = processGapsBatch;