"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleBackfill = scheduleBackfill;
var _pMap = _interopRequireDefault(require("p-map"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _esQuery = require("@kbn/es-query");
var _saved_objects = require("../../../../saved_objects");
var _rule = require("../../../../data/rule");
var _constants = require("../../../../rules_client/common/constants");
var _lib = require("../../../../lib");
var _authorization = require("../../../../authorization");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _schemas = require("./schemas");
var _transforms = require("../../../rule/transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function scheduleBackfill(context, params, gaps) {
  var _aggregations$alertTy;
  try {
    _schemas.scheduleBackfillParamsSchema.validate(params);
  } catch (error) {
    throw _boom.default.badRequest(`Error validating backfill schedule parameters "${JSON.stringify(params)}" - ${error.message}`);
  }

  // Get rule SO IDs
  const ruleIds = params.map(param => param.ruleId);
  const kueryNodeFilter = (0, _lib.convertRuleIdsToKueryNode)(ruleIds);
  let authorizationTuple;
  try {
    authorizationTuple = await context.authorization.getFindAuthorizationFilter({
      authorizationEntity: _authorization.AlertingAuthorizationEntity.Rule,
      filterOpts: _constants.alertingAuthorizationFilterOpts
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.SCHEDULE_BACKFILL,
      error
    }));
    throw error;
  }
  const {
    filter: authorizationFilter
  } = authorizationTuple;
  const kueryNodeFilterWithAuth = authorizationFilter && kueryNodeFilter ? _esQuery.nodeBuilder.and([kueryNodeFilter, authorizationFilter]) : kueryNodeFilter;
  const {
    aggregations
  } = await (0, _rule.findRulesSo)({
    savedObjectsClient: context.unsecuredSavedObjectsClient,
    savedObjectsFindOptions: {
      filter: kueryNodeFilterWithAuth,
      page: 1,
      perPage: 0,
      ...(context.namespace ? {
        namespaces: [context.namespace]
      } : undefined),
      aggs: {
        alertTypeId: {
          multi_terms: {
            terms: [{
              field: 'alert.attributes.alertTypeId'
            }, {
              field: 'alert.attributes.consumer'
            }]
          }
        }
      }
    }
  });
  const buckets = aggregations === null || aggregations === void 0 ? void 0 : (_aggregations$alertTy = aggregations.alertTypeId) === null || _aggregations$alertTy === void 0 ? void 0 : _aggregations$alertTy.buckets;
  if (buckets === undefined || !buckets.length) {
    throw _boom.default.badRequest(`No rules matching ids ${ruleIds} found to schedule backfill`);
  }
  await (0, _pMap.default)(buckets, async ({
    key: [ruleType, consumer]
  }) => {
    context.ruleTypeRegistry.ensureRuleTypeEnabled(ruleType);
    try {
      await context.authorization.ensureAuthorized({
        ruleTypeId: ruleType,
        consumer,
        operation: _authorization.WriteOperations.ScheduleBackfill,
        entity: _authorization.AlertingAuthorizationEntity.Rule
      });
    } catch (error) {
      var _context$auditLogger2;
      (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
        action: _audit_events.RuleAuditAction.SCHEDULE_BACKFILL,
        error
      }));
      throw error;
    }
  }, {
    concurrency: _constants.RULE_TYPE_CHECKS_CONCURRENCY
  });
  const rulesFinder = await context.encryptedSavedObjectsClient.createPointInTimeFinderDecryptedAsInternalUser({
    filter: kueryNodeFilterWithAuth,
    type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
    perPage: 100,
    ...(context.namespace ? {
      namespaces: [context.namespace]
    } : undefined)
  });
  let rulesToSchedule = [];
  for await (const response of rulesFinder.find()) {
    for (const rule of response.saved_objects) {
      var _context$auditLogger3;
      (_context$auditLogger3 = context.auditLogger) === null || _context$auditLogger3 === void 0 ? void 0 : _context$auditLogger3.log((0, _audit_events.ruleAuditEvent)({
        action: _audit_events.RuleAuditAction.SCHEDULE_BACKFILL,
        savedObject: {
          type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
          id: rule.id,
          name: rule.attributes.name
        }
      }));
    }
    rulesToSchedule = [...response.saved_objects];
  }
  const actionsClient = await context.getActionsClient();
  return await context.backfillClient.bulkQueue({
    actionsClient,
    auditLogger: context.auditLogger,
    params,
    rules: rulesToSchedule.map(({
      id,
      attributes,
      references
    }) => {
      const ruleType = context.ruleTypeRegistry.get(attributes.alertTypeId);
      return (0, _transforms.transformRuleAttributesToRuleDomain)(attributes, {
        id,
        logger: context.logger,
        ruleType,
        references,
        omitGeneratedValues: false
      }, connectorId => actionsClient.isSystemAction(connectorId));
    }),
    gaps,
    ruleTypeRegistry: context.ruleTypeRegistry,
    spaceId: context.spaceId,
    unsecuredSavedObjectsClient: context.unsecuredSavedObjectsClient,
    eventLogClient: await context.getEventLogClient(),
    internalSavedObjectsRepository: context.internalSavedObjectsRepository,
    eventLogger: context.eventLogger
  });
}