"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTokenCountFromOpenAIStream = getTokenCountFromOpenAIStream;
var _gptTokenizer = require("gpt-tokenizer");
var _lodash = require("lodash");
var _promises = require("stream/promises");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getTokenCountFromOpenAIStream({
  responseStream,
  body,
  logger
}) {
  let responseBody = '';
  responseStream.on('data', chunk => {
    responseBody += chunk.toString();
  });
  try {
    await (0, _promises.finished)(responseStream);
  } catch (e) {
    logger.error('An error occurred while calculating streaming response tokens');
  }
  let completionUsage;
  const response = responseBody.split('\n').filter(line => {
    return line.startsWith('data: ') && !line.endsWith('[DONE]');
  }).map(line => {
    return JSON.parse(line.replace('data: ', ''));
  }).filter(line => {
    return 'object' in line && line.object === 'chat.completion.chunk';
  }).reduce((prev, line) => {
    var _line$choices;
    if (line.usage) {
      completionUsage = line.usage;
    }
    if ((_line$choices = line.choices) !== null && _line$choices !== void 0 && _line$choices.length) {
      var _msg$function_call, _msg$function_call2;
      const msg = line.choices[0].delta;
      prev.content += msg.content || '';
      prev.function_call.name += ((_msg$function_call = msg.function_call) === null || _msg$function_call === void 0 ? void 0 : _msg$function_call.name) || '';
      prev.function_call.arguments += ((_msg$function_call2 = msg.function_call) === null || _msg$function_call2 === void 0 ? void 0 : _msg$function_call2.arguments) || '';
    }
    return prev;
  }, {
    content: '',
    function_call: {
      name: '',
      arguments: ''
    }
  });

  // not all openAI compatible providers emit completion chunk, so we still have to support
  // manually counting the tokens
  if (completionUsage) {
    return {
      prompt: completionUsage.prompt_tokens,
      completion: completionUsage.completion_tokens,
      total: completionUsage.total_tokens
    };
  } else {
    const promptTokens = manuallyCountPromptTokens(body);
    const completionTokens = manuallyCountCompletionTokens(response);
    return {
      prompt: promptTokens,
      completion: completionTokens,
      total: promptTokens + completionTokens
    };
  }
}
const manuallyCountCompletionTokens = response => {
  return (0, _gptTokenizer.encode)(JSON.stringify((0, _lodash.omitBy)({
    content: response.content || undefined,
    function_call: response.function_call.name ? response.function_call : undefined
  }, _lodash.isEmpty))).length;
};
const manuallyCountPromptTokens = requestBody => {
  var _chatCompletionReques;
  const chatCompletionRequest = JSON.parse(requestBody);

  // per https://github.com/openai/openai-cookbook/blob/main/examples/How_to_count_tokens_with_tiktoken.ipynb
  const tokensFromMessages = (0, _gptTokenizer.encode)(chatCompletionRequest.messages.map(msg => `<|start|>${msg.role}\n${msg.content}\n${'name' in msg ? msg.name : 'function_call' in msg && msg.function_call ? msg.function_call.name + '\n' + msg.function_call.arguments : ''}<|end|>`).join('\n')).length;

  // this is an approximation. OpenAI cuts off a function schema
  // at a certain level of nesting, so their token count might
  // be lower than what we are calculating here.

  const tokensFromFunctions = chatCompletionRequest.functions ? (0, _gptTokenizer.encode)((_chatCompletionReques = chatCompletionRequest.functions) === null || _chatCompletionReques === void 0 ? void 0 : _chatCompletionReques.map(fn => `<|start|>${fn.name}\n${fn.description}\n${JSON.stringify(fn.parameters)}<|end|>`).join('\n')).length : 0;
  return tokensFromMessages + tokensFromFunctions;
};