"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PHPExporter = void 0;
const fs_1 = require("fs");
const path_1 = __importDefault(require("path"));
const handlebars_1 = __importDefault(require("handlebars"));
require("./templates");
function isSupportedAPI(req) {
    const unsupportedApis = [
        "_internal.prevalidate_node_removal",
        "indices.create_from",
        "indices.migrate_reindex",
        "indices.get_migrate_reindex_status",
        "indices.cancel_migrate_reindex",
        "security.delegate_pki",
    ];
    if (req.api && unsupportedApis.includes(req.api)) {
        return false;
    }
    let supported = false;
    if (req.availability &&
        (req.availability.stack?.visibility !== "private" ||
            req.availability.stack?.stability !== "experimental" ||
            (req.availability.serverless &&
                req.availability.serverless.visibility !== "private"))) {
        supported = true;
    }
    return supported;
}
class PHPExporter {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async check(requests) {
        // only return true if all requests are for Elasticsearch
        return requests
            .map((req) => req.service == "es")
            .reduce((prev, curr) => prev && curr, true);
    }
    async convert(requests, options) {
        if (!(await this.check(requests))) {
            throw new Error("Cannot perform conversion");
        }
        return (await this.getTemplate())({ requests, ...options });
    }
    phpprint(data, indent, startIndented) {
        // render a JSON data structure with PHP syntax
        if (typeof data === "number" || typeof data === "boolean") {
            return (startIndented ? indent : "") + String(data);
        }
        else if (data === null || data === undefined) {
            return (startIndented ? indent : "") + "null";
        }
        else if (typeof data === "string") {
            return ((startIndented ? indent : "") +
                JSON.stringify(data).replaceAll("${", "\\${"));
        }
        else if (Array.isArray(data)) {
            const elements = data.length === 0
                ? ""
                : data
                    .map((elem) => "    " + this.phpprint(elem, indent + "    ", true))
                    .join(",\n") + ",\n";
            return ((startIndented ? indent + "    " : "") +
                "array(\n" +
                elements +
                indent +
                "    )");
        }
        else if (typeof data === "object") {
            const elements = Object.keys(data).length === 0
                ? ""
                : Object.keys(data)
                    .map((key) => `${indent}        "${key}" => ` +
                    this.phpprint(data[key], indent + "    ", false))
                    .join(",\n") + ",\n";
            if (elements.length) {
                return ((startIndented ? indent : "") + "[\n" + elements + indent + "    ]");
            }
            else {
                return (startIndented ? indent : "") + "new ArrayObject([])";
            }
        }
        else {
            throw new Error(`Unexpected type in JSON payload`);
        }
    }
    async getTemplate() {
        if (!this.template) {
            // custom data renderer for Python
            handlebars_1.default.registerHelper("phpprint", (context) => {
                return this.phpprint(context, "", false);
            });
            //
            handlebars_1.default.registerHelper("needsRequestFactory", function (options) {
                let anyUnsupported = false;
                for (const request of this.requests) {
                    if (!isSupportedAPI(request)) {
                        anyUnsupported = true;
                        break;
                    }
                }
                if (anyUnsupported) {
                    return options.fn(this);
                }
                else {
                    return options.inverse(this);
                }
            });
            // custom conditional for requests without any arguments
            handlebars_1.default.registerHelper("hasArgs", function (options) {
                if (Object.keys(this.params ?? {}).length +
                    Object.keys(this.query ?? {}).length +
                    Object.keys(this.body ?? {}).length >
                    0) {
                    return options.fn(this);
                }
                else {
                    return options.inverse(this);
                }
            });
            // custom conditional to separate supported vs unsupported APIs
            handlebars_1.default.registerHelper("supportedApi", function (options) {
                if (isSupportedAPI(this)) {
                    return options.fn(this);
                }
                else {
                    return options.inverse(this);
                }
            });
            handlebars_1.default.registerHelper("phpEndpoint", (name) => {
                const snakeToCamel = (str) => str
                    .toLowerCase()
                    .replace(/([-_][a-z])/g, (group) => group.toUpperCase().replace("-", "").replace("_", ""));
                const parts = name.split(".").map((part) => snakeToCamel(part));
                const phpParts = parts.slice(0, -1).map((part) => part + "()");
                phpParts.push(parts.slice(-1));
                return phpParts.join("->");
            });
            if (process.env.NODE_ENV !== "test") {
                this.template = handlebars_1.default.templates["php.tpl"];
            }
            else {
                // when running tests we read the templates directly, in case the
                // compiled file is not up to date
                const t = (0, fs_1.readFileSync)(path_1.default.join(__dirname, "./php.tpl"), "utf-8");
                this.template = handlebars_1.default.compile(t);
            }
        }
        return this.template;
    }
}
exports.PHPExporter = PHPExporter;
