/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.Build;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.OriginalIndices;
import org.elasticsearch.action.support.CountDownActionListener;
import org.elasticsearch.action.support.IndicesOptions;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.action.support.RefCountingListener;
import org.elasticsearch.action.support.RefCountingRunnable;
import org.elasticsearch.client.internal.RemoteClusterClient;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.project.ProjectResolver;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.indices.IndicesExpressionGrouper;
import org.elasticsearch.node.ReportingService;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.LinkedProjectConfig;
import org.elasticsearch.transport.NoSuchRemoteClusterException;
import org.elasticsearch.transport.RemoteClusterAware;
import org.elasticsearch.transport.RemoteClusterAwareClient;
import org.elasticsearch.transport.RemoteClusterConnection;
import org.elasticsearch.transport.RemoteClusterCredentialsManager;
import org.elasticsearch.transport.RemoteClusterPortSettings;
import org.elasticsearch.transport.RemoteClusterServerInfo;
import org.elasticsearch.transport.RemoteClusterSettings;
import org.elasticsearch.transport.RemoteConnectionInfo;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportService;

public final class RemoteClusterService
extends RemoteClusterAware
implements Closeable,
ReportingService<RemoteClusterServerInfo>,
IndicesExpressionGrouper {
    private static final Logger logger = LogManager.getLogger(RemoteClusterService.class);
    public static final String REMOTE_CLUSTER_HANDSHAKE_ACTION_NAME = "cluster:internal/remote_cluster/handshake";
    private final boolean isRemoteClusterClient;
    private final boolean isSearchNode;
    private final boolean isStateless;
    private final boolean remoteClusterServerEnabled;
    private final TransportService transportService;
    private final Map<ProjectId, Map<String, RemoteClusterConnection>> remoteClusters;
    private final RemoteClusterCredentialsManager remoteClusterCredentialsManager;
    private final ProjectResolver projectResolver;
    private final boolean crossProjectEnabled;

    public boolean isRemoteClusterServerEnabled() {
        return this.remoteClusterServerEnabled;
    }

    RemoteClusterService(Settings settings, TransportService transportService, ProjectResolver projectResolver) {
        super(settings);
        this.isRemoteClusterClient = DiscoveryNode.isRemoteClusterClient(settings);
        this.isSearchNode = DiscoveryNode.hasRole(settings, DiscoveryNodeRole.SEARCH_ROLE);
        this.isStateless = DiscoveryNode.isStateless(settings);
        this.remoteClusterServerEnabled = RemoteClusterPortSettings.REMOTE_CLUSTER_SERVER_ENABLED.get(settings);
        this.transportService = transportService;
        this.projectResolver = projectResolver;
        this.remoteClusters = projectResolver.supportsMultipleProjects() ? ConcurrentCollections.newConcurrentMap() : Map.of(ProjectId.DEFAULT, ConcurrentCollections.newConcurrentMap());
        this.remoteClusterCredentialsManager = new RemoteClusterCredentialsManager(settings);
        if (this.remoteClusterServerEnabled) {
            RemoteClusterService.registerRemoteClusterHandshakeRequestHandler(transportService);
        }
        this.crossProjectEnabled = settings.getAsBoolean("serverless.cross_project.enabled", false);
    }

    public Map<String, OriginalIndices> groupIndices(Set<String> remoteClusterNames, IndicesOptions indicesOptions, String[] indices, boolean returnLocalAll) {
        HashMap<String, OriginalIndices> originalIndicesMap = new HashMap<String, OriginalIndices>();
        Map<Object, Object> groupedIndices = !returnLocalAll && IndexNameExpressionResolver.isNoneExpression(indices) ? Map.of() : this.groupClusterIndices(remoteClusterNames, indices);
        if (groupedIndices.isEmpty()) {
            if (returnLocalAll) {
                originalIndicesMap.put("", new OriginalIndices(Strings.EMPTY_ARRAY, indicesOptions));
            }
        } else {
            for (Map.Entry<Object, Object> entry : groupedIndices.entrySet()) {
                String clusterAlias = (String)entry.getKey();
                List originalIndices = (List)entry.getValue();
                originalIndicesMap.put(clusterAlias, new OriginalIndices(originalIndices.toArray(new String[0]), indicesOptions));
            }
        }
        return originalIndicesMap;
    }

    public Map<String, OriginalIndices> groupIndices(Set<String> remoteClusterNames, IndicesOptions indicesOptions, String[] indices) {
        return this.groupIndices(remoteClusterNames, indicesOptions, indices, true);
    }

    @Override
    public Map<String, OriginalIndices> groupIndices(IndicesOptions indicesOptions, String[] indices, boolean returnLocalAll) {
        return this.groupIndices(this.getRegisteredRemoteClusterNames(), indicesOptions, indices, returnLocalAll);
    }

    public Map<String, OriginalIndices> groupIndices(IndicesOptions indicesOptions, String[] indices) {
        return this.groupIndices(this.getRegisteredRemoteClusterNames(), indicesOptions, indices, true);
    }

    public Set<String> getRegisteredRemoteClusterNames() {
        return this.getConnectionsMapForCurrentProject().keySet();
    }

    public Transport.Connection getConnection(DiscoveryNode node, String cluster) {
        return this.getRemoteClusterConnection(cluster).getConnection(node);
    }

    void ensureConnected(String clusterAlias, ActionListener<Void> listener) {
        RemoteClusterConnection remoteClusterConnection;
        try {
            remoteClusterConnection = this.getRemoteClusterConnection(clusterAlias);
        }
        catch (NoSuchRemoteClusterException e) {
            listener.onFailure(e);
            return;
        }
        remoteClusterConnection.ensureConnected(listener);
    }

    public Optional<Boolean> isSkipUnavailable(String clusterAlias) {
        if (this.crossProjectEnabled) {
            return Optional.empty();
        }
        return Optional.of(this.getRemoteClusterConnection(clusterAlias).isSkipUnavailable());
    }

    public boolean crossProjectEnabled() {
        return this.crossProjectEnabled;
    }

    public boolean shouldSkipOnFailure(String clusterAlias, Boolean allowPartialSearchResults) {
        return this.isSkipUnavailable(clusterAlias).orElseGet(() -> allowPartialSearchResults != null && allowPartialSearchResults != false);
    }

    public Transport.Connection getConnection(String cluster) {
        return this.getRemoteClusterConnection(cluster).getConnection();
    }

    public void maybeEnsureConnectedAndGetConnection(String clusterAlias, boolean ensureConnected, ActionListener<Transport.Connection> listener) {
        ActionListener<Void> ensureConnectedListener = listener.delegateFailureAndWrap((l, nullValue) -> ActionListener.completeWith(l, () -> {
            try {
                return this.getConnection(clusterAlias);
            }
            catch (ConnectTransportException e) {
                if (!ensureConnected) {
                    this.ensureConnected(clusterAlias, ActionListener.noop());
                }
                throw e;
            }
        }));
        if (ensureConnected) {
            this.ensureConnected(clusterAlias, ensureConnectedListener);
        } else {
            ensureConnectedListener.onResponse(null);
        }
    }

    public RemoteClusterConnection getRemoteClusterConnection(String cluster) {
        this.ensureClientIsEnabled();
        RemoteClusterConnection connection = this.getConnectionsMapForCurrentProject().get(cluster);
        if (connection == null) {
            throw new NoSuchRemoteClusterException(cluster);
        }
        return connection;
    }

    @Override
    public void skipUnavailableChanged(ProjectId originProjectId, ProjectId linkedProjectId, String linkedProjectAlias, boolean skipUnavailable) {
        RemoteClusterConnection remote = this.getConnectionsMapForProject(originProjectId).get(linkedProjectAlias);
        if (remote != null) {
            remote.setSkipUnavailable(skipUnavailable);
        }
    }

    public synchronized void updateRemoteClusterCredentials(Supplier<Settings> settingsSupplier, ActionListener<Void> listener) {
        ProjectId projectId = this.projectResolver.getProjectId();
        Settings settings = settingsSupplier.get();
        RemoteClusterCredentialsManager.UpdateRemoteClusterCredentialsResult result = this.remoteClusterCredentialsManager.updateClusterCredentials(settings);
        int totalConnectionsToRebuild = result.addedClusterAliases().size() + result.removedClusterAliases().size();
        if (totalConnectionsToRebuild == 0) {
            logger.debug("project [{}] no connection rebuilding required after credentials update", (Object)projectId);
            listener.onResponse(null);
            return;
        }
        logger.info("project [{}] rebuilding [{}] connections after credentials update", (Object)projectId, (Object)totalConnectionsToRebuild);
        try (RefCountingRunnable connectionRefs = new RefCountingRunnable(() -> listener.onResponse(null));){
            for (String clusterAlias : result.addedClusterAliases()) {
                this.maybeRebuildConnectionOnCredentialsChange(projectId, clusterAlias, settings, connectionRefs);
            }
            for (String clusterAlias : result.removedClusterAliases()) {
                this.maybeRebuildConnectionOnCredentialsChange(projectId, clusterAlias, settings, connectionRefs);
            }
        }
    }

    private void maybeRebuildConnectionOnCredentialsChange(final ProjectId projectId, final String clusterAlias, Settings newSettings, RefCountingRunnable connectionRefs) {
        Map<String, RemoteClusterConnection> connectionsMap = this.getConnectionsMapForProject(projectId);
        if (!connectionsMap.containsKey(clusterAlias)) {
            logger.info("project [{}] no connection rebuild required for remote cluster [{}] after credentials change", (Object)projectId, (Object)clusterAlias);
            return;
        }
        Settings mergedSettings = Settings.builder().put(this.settings, false).put(newSettings, false).build();
        LinkedProjectConfig config = RemoteClusterSettings.toConfig(projectId, ProjectId.DEFAULT, clusterAlias, mergedSettings);
        this.updateRemoteCluster(config, true, ActionListener.releaseAfter(new ActionListener<RemoteClusterConnectionStatus>(this){

            @Override
            public void onResponse(RemoteClusterConnectionStatus status) {
                logger.info("project [{}] remote cluster connection [{}] updated after credentials change: [{}]", (Object)projectId, (Object)clusterAlias, (Object)status);
            }

            @Override
            public void onFailure(Exception e) {
                logger.warn(() -> "project [" + String.valueOf(projectId) + "] failed to update remote cluster connection [" + clusterAlias + "] after credentials change", (Throwable)e);
            }
        }, connectionRefs.acquire()));
    }

    @Override
    public void updateLinkedProject(LinkedProjectConfig config) {
        final ProjectId projectId = config.originProjectId();
        final String clusterAlias = config.linkedProjectAlias();
        CountDownLatch latch = new CountDownLatch(1);
        this.updateRemoteCluster(config, false, ActionListener.runAfter(new ActionListener<RemoteClusterConnectionStatus>(this){

            @Override
            public void onResponse(RemoteClusterConnectionStatus status) {
                logger.info("project [{}] remote cluster connection [{}] updated: {}", (Object)projectId, (Object)clusterAlias, (Object)status);
            }

            @Override
            public void onFailure(Exception e) {
                logger.warn(() -> "project [" + String.valueOf(projectId) + " failed to update remote cluster connection [" + clusterAlias + "]", (Throwable)e);
            }
        }, latch::countDown));
        try {
            if (!latch.await(10L, TimeUnit.SECONDS)) {
                logger.warn("project [{}] failed to update remote cluster connection [{}] within {}", (Object)projectId, (Object)clusterAlias, (Object)TimeValue.timeValueSeconds((long)10L));
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    synchronized void updateRemoteCluster(LinkedProjectConfig config, boolean forceRebuild, ActionListener<RemoteClusterConnectionStatus> listener) {
        ProjectId projectId = config.originProjectId();
        String clusterAlias = config.linkedProjectAlias();
        Map<String, RemoteClusterConnection> connectionMap = this.getConnectionsMapForProject(projectId);
        RemoteClusterConnection remote = connectionMap.get(clusterAlias);
        if (!config.isConnectionEnabled()) {
            try {
                IOUtils.close((Closeable)remote);
            }
            catch (IOException e) {
                logger.warn("project [" + String.valueOf(projectId) + "] failed to close remote cluster connections for cluster: " + clusterAlias, (Throwable)e);
            }
            connectionMap.remove(clusterAlias);
            listener.onResponse(RemoteClusterConnectionStatus.DISCONNECTED);
            return;
        }
        if (remote == null) {
            remote = new RemoteClusterConnection(config, this.transportService, this.remoteClusterCredentialsManager);
            connectionMap.put(clusterAlias, remote);
            remote.ensureConnected(listener.map(ignored -> RemoteClusterConnectionStatus.CONNECTED));
        } else if (forceRebuild || remote.shouldRebuildConnection(config)) {
            try {
                IOUtils.close((Closeable)remote);
            }
            catch (IOException e) {
                logger.warn("project [" + String.valueOf(projectId) + "] failed to close remote cluster connections for cluster: " + clusterAlias, (Throwable)e);
            }
            connectionMap.remove(clusterAlias);
            remote = new RemoteClusterConnection(config, this.transportService, this.remoteClusterCredentialsManager);
            connectionMap.put(clusterAlias, remote);
            remote.ensureConnected(listener.map(ignored -> RemoteClusterConnectionStatus.RECONNECTED));
        } else {
            listener.onResponse(RemoteClusterConnectionStatus.UNCHANGED);
        }
    }

    void initializeRemoteClusters(Collection<LinkedProjectConfig> configs) {
        if (configs.isEmpty()) {
            return;
        }
        ProjectId projectId = this.projectResolver.getProjectId();
        TimeValue timeValue = RemoteClusterSettings.REMOTE_INITIAL_CONNECTION_TIMEOUT_SETTING.get(this.settings);
        PlainActionFuture<Void> future = new PlainActionFuture<Void>();
        CountDownActionListener listener = new CountDownActionListener(configs.size(), future);
        for (LinkedProjectConfig config : configs) {
            this.updateRemoteCluster(config, false, listener.map(ignored -> null));
        }
        try {
            future.get(timeValue.millis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (TimeoutException ex) {
            logger.warn("project [{}] failed to connect to remote clusters within {}", (Object)projectId, (Object)timeValue.toString());
        }
        catch (Exception e) {
            logger.warn("project [" + String.valueOf(projectId) + "] failed to connect to remote clusters", (Throwable)e);
        }
    }

    @Override
    public void close() throws IOException {
        IOUtils.close((Iterable)this.remoteClusters.values().stream().flatMap(map -> map.values().stream()).collect(Collectors.toList()));
    }

    public Stream<RemoteConnectionInfo> getRemoteConnectionInfos() {
        return this.getConnectionsMapForCurrentProject().values().stream().map(RemoteClusterConnection::getConnectionInfo);
    }

    @Override
    public RemoteClusterServerInfo info() {
        if (this.remoteClusterServerEnabled) {
            return new RemoteClusterServerInfo(this.transportService.boundRemoteAccessAddress());
        }
        return null;
    }

    public void collectNodes(Set<String> clusters, ActionListener<BiFunction<String, String, DiscoveryNode>> listener) {
        this.ensureClientIsEnabled();
        Map<String, RemoteClusterConnection> projectConnectionsMap = this.getConnectionsMapForCurrentProject();
        HashMap<String, RemoteClusterConnection> connectionsMap = new HashMap<String, RemoteClusterConnection>();
        for (String cluster2 : clusters) {
            RemoteClusterConnection connection2 = projectConnectionsMap.get(cluster2);
            if (connection2 == null) {
                listener.onFailure(new NoSuchRemoteClusterException(cluster2));
                return;
            }
            connectionsMap.put(cluster2, connection2);
        }
        HashMap clusterMap = new HashMap();
        ActionListener<Void> finalListener = listener.safeMap(ignored -> (clusterAlias, nodeId) -> clusterMap.getOrDefault(clusterAlias, s -> null).apply((String)nodeId));
        try (RefCountingListener refs = new RefCountingListener(finalListener);){
            connectionsMap.forEach((cluster, connection) -> connection.collectNodes(refs.acquire(nodeLookup -> {
                Map map = clusterMap;
                synchronized (map) {
                    clusterMap.put(cluster, nodeLookup);
                }
            })));
        }
    }

    public RemoteClusterClient getRemoteClusterClient(String clusterAlias, Executor responseExecutor, DisconnectedStrategy disconnectedStrategy) {
        this.ensureClientIsEnabled();
        if (!this.transportService.getRemoteClusterService().getRegisteredRemoteClusterNames().contains(clusterAlias)) {
            throw new NoSuchRemoteClusterException(clusterAlias);
        }
        return new RemoteClusterAwareClient(this.transportService, clusterAlias, responseExecutor, switch (disconnectedStrategy.ordinal()) {
            default -> throw new MatchException(null, null);
            case 0 -> true;
            case 1 -> false;
            case 2 -> this.transportService.getRemoteClusterService().isSkipUnavailable(clusterAlias).orElse(true) == false;
        });
    }

    public void ensureClientIsEnabled() {
        if (this.isRemoteClusterClient) {
            return;
        }
        if (!this.isStateless) {
            throw new IllegalArgumentException("node [" + this.getNodeName() + "] does not have the [" + DiscoveryNodeRole.REMOTE_CLUSTER_CLIENT_ROLE.roleName() + "] role");
        }
        if (!this.isSearchNode) {
            throw new IllegalArgumentException("node [" + this.getNodeName() + "] must have the [" + DiscoveryNodeRole.REMOTE_CLUSTER_CLIENT_ROLE.roleName() + "] role or the [" + DiscoveryNodeRole.SEARCH_ROLE.roleName() + "] role in stateless environments to use linked project client features");
        }
    }

    static void registerRemoteClusterHandshakeRequestHandler(TransportService transportService) {
        transportService.registerRequestHandler(REMOTE_CLUSTER_HANDSHAKE_ACTION_NAME, EsExecutors.DIRECT_EXECUTOR_SERVICE, false, false, TransportService.HandshakeRequest::new, (request, channel, task) -> {
            if (!"_remote_cluster".equals(channel.getProfileName())) {
                throw new IllegalArgumentException(Strings.format("remote cluster handshake action requires channel profile to be [%s], but got [%s]", "_remote_cluster", channel.getProfileName()));
            }
            logger.trace("handling remote cluster handshake request");
            channel.sendResponse(new TransportService.HandshakeResponse(transportService.getLocalNode().getVersion(), Build.current().hash(), transportService.getLocalNode().withTransportAddress(transportService.boundRemoteAccessAddress().publishAddress()), transportService.clusterName));
        });
    }

    private Map<String, RemoteClusterConnection> getConnectionsMapForCurrentProject() {
        return this.getConnectionsMapForProject(this.projectResolver.getProjectId());
    }

    private Map<String, RemoteClusterConnection> getConnectionsMapForProject(ProjectId projectId) {
        if (this.projectResolver.supportsMultipleProjects()) {
            return this.remoteClusters.computeIfAbsent(projectId, unused -> ConcurrentCollections.newConcurrentMap());
        }
        assert (ProjectId.DEFAULT.equals(projectId)) : "Only the default project ID should be used when multiple projects are not supported";
        return this.remoteClusters.get(projectId);
    }

    static enum RemoteClusterConnectionStatus {
        CONNECTED,
        DISCONNECTED,
        RECONNECTED,
        UNCHANGED;

    }

    public static enum DisconnectedStrategy {
        RECONNECT_IF_DISCONNECTED,
        FAIL_IF_DISCONNECTED,
        RECONNECT_UNLESS_SKIP_UNAVAILABLE;

    }
}

