"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initTracing = initTracing;
var _sdk = require("@elastic/opentelemetry-node/sdk");
var _inferenceTracing = require("@kbn/inference-tracing");
var _std = require("@kbn/std");
var _api = require("@opentelemetry/api");
var _contextAsyncHooks = require("@opentelemetry/context-async-hooks");
var _lodash = require("lodash");
var _cleanupBeforeExit = require("@kbn/cleanup-before-exit");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Initialize the OpenTelemetry tracing provider
 * @param resource The OpenTelemetry resource information
 * @param tracingConfig The OpenTelemetry tracing configuration
 */
function initTracing({
  resource,
  tracingConfig
}) {
  const contextManager = new _contextAsyncHooks.AsyncLocalStorageContextManager();
  _api.context.setGlobalContextManager(contextManager);
  contextManager.enable();

  // this is used for late-binding of span processors
  const lateBindingProcessor = _.LateBindingSpanProcessor.get();
  const allSpanProcessors = [lateBindingProcessor];
  _api.propagation.setGlobalPropagator(new _sdk.core.CompositePropagator({
    propagators: [new _sdk.core.W3CTraceContextPropagator(), new _sdk.core.W3CBaggagePropagator()]
  }));
  const traceIdSampler = new _sdk.tracing.TraceIdRatioBasedSampler(tracingConfig.sample_rate);
  const nodeTracerProvider = new _sdk.node.NodeTracerProvider({
    // by default, base sampling on parent context,
    // or for root spans, based on the configured sample rate
    sampler: new _sdk.tracing.ParentBasedSampler({
      root: traceIdSampler
    }),
    spanProcessors: allSpanProcessors,
    resource
  });
  (0, _lodash.castArray)(tracingConfig.exporters).forEach(exporter => {
    const variant = (0, _std.fromExternalVariant)(exporter);
    switch (variant.type) {
      case 'langfuse':
        _.LateBindingSpanProcessor.get().register(new _inferenceTracing.LangfuseSpanProcessor(variant.value));
        break;
      case 'phoenix':
        _.LateBindingSpanProcessor.get().register(new _inferenceTracing.PhoenixSpanProcessor(variant.value));
        break;
    }
  });
  _api.trace.setGlobalTracerProvider(nodeTracerProvider);
  _api.propagation.setGlobalPropagator(new _sdk.core.CompositePropagator({
    propagators: [new _sdk.core.W3CTraceContextPropagator(), new _sdk.core.W3CBaggagePropagator()]
  }));
  const shutdown = async () => {
    await Promise.all(allSpanProcessors.map(processor => processor.shutdown()));
  };
  (0, _cleanupBeforeExit.cleanupBeforeExit)(() => shutdown(), {
    blockExit: true,
    timeout: 30_000
  });
}