"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initTelemetry = void 0;
var _apmConfigLoader = require("@kbn/apm-config-loader");
var _tracing = require("@kbn/tracing");
var _metrics = require("@kbn/metrics");
var _sdk = require("@elastic/opentelemetry-node/sdk");
var _semanticConventions = require("@opentelemetry/semantic-conventions");
var _incubating = require("@opentelemetry/semantic-conventions/incubating");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 *
 * Initializes OpenTelemetry (currently only tracing)
 *
 * @param argv                Process arguments
 * @param rootDir             Root dir of Kibana repo
 * @param isDistributable     Whether this is a distributable build
 * @param serviceName         The service name used in resource attributes
 * @returns                   A function that can be called on shutdown and allows exporters to flush their queue.
 */
const initTelemetry = (argv, rootDir, isDistributable, serviceName) => {
  const apmConfigLoader = (0, _apmConfigLoader.loadConfiguration)(argv, rootDir, isDistributable);
  const apmConfig = apmConfigLoader.getConfig(serviceName);
  const telemetryConfig = apmConfigLoader.getTelemetryConfig();
  const monitoringCollectionConfig = apmConfigLoader.getMonitoringCollectionConfig();

  // attributes.resource.*
  const resource = _sdk.resources.resourceFromAttributes({
    [_semanticConventions.ATTR_SERVICE_NAME]: apmConfig.serviceName,
    [_semanticConventions.ATTR_SERVICE_VERSION]: apmConfig.serviceVersion,
    [_incubating.ATTR_SERVICE_INSTANCE_ID]: apmConfig.serviceNodeName,
    // Reverse-mapping APM Server transformations:
    // https://github.com/elastic/apm-data/blob/2f9cdbf722e5be5bf77d99fbcaab7a70a7e83fff/input/otlp/metadata.go#L69-L74
    [_incubating.ATTR_DEPLOYMENT_ENVIRONMENT_NAME]: apmConfig.environment,
    // From https://opentelemetry.io/docs/specs/semconv/resource/process/
    'process.pid': process.pid,
    'process.runtime.name': 'nodejs',
    'process.runtime.version': process.version,
    ...apmConfig.globalLabels
  });
  if (telemetryConfig.enabled) {
    if (telemetryConfig.tracing.enabled) {
      (0, _tracing.initTracing)({
        resource,
        tracingConfig: telemetryConfig.tracing
      });
    }
    if (telemetryConfig.metrics.enabled || monitoringCollectionConfig.enabled) {
      (0, _metrics.initMetrics)({
        resource,
        metricsConfig: telemetryConfig.metrics,
        monitoringCollectionConfig
      });
    }
  }
};
exports.initTelemetry = initTelemetry;