"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runOnceSyntheticsMonitorRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _lodash = require("lodash");
var _utils = require("../monitor_cruds/add_monitor/utils");
var _runtime_types = require("../../../common/runtime_types");
var _constants = require("../../../common/constants");
var _monitor_validation = require("../monitor_cruds/monitor_validation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const runOnceSyntheticsMonitorRoute = () => ({
  method: 'POST',
  path: _constants.SYNTHETICS_API_URLS.RUN_ONCE_MONITOR + '/{monitorId}',
  validate: {
    body: _configSchema.schema.any(),
    params: _configSchema.schema.object({
      monitorId: _configSchema.schema.string({
        minLength: 1,
        maxLength: 1024
      })
    })
  },
  handler: async ({
    request,
    response,
    syntheticsMonitorClient,
    savedObjectsClient,
    spaceId
  }) => {
    const monitor = request.body;
    const {
      monitorId
    } = request.params;
    if ((0, _lodash.isEmpty)(monitor)) {
      return response.badRequest({
        body: {
          message: 'Monitor data is empty.'
        }
      });
    }
    const validationResult = (0, _monitor_validation.validateMonitor)(monitor, spaceId);
    const decodedMonitor = validationResult.decodedMonitor;
    if (!validationResult.valid || !decodedMonitor) {
      const {
        reason: message,
        details,
        payload
      } = validationResult;
      return response.badRequest({
        body: {
          message,
          attributes: {
            details,
            ...payload
          }
        }
      });
    }
    const privateLocations = await (0, _utils.getPrivateLocationsForMonitor)(savedObjectsClient, decodedMonitor);
    const [, errors] = await syntheticsMonitorClient.testNowConfigs({
      monitor: {
        ...decodedMonitor,
        [_runtime_types.ConfigKey.CONFIG_ID]: monitorId,
        [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: monitorId
      },
      id: monitorId,
      testRunId: monitorId
    }, privateLocations, spaceId, true);
    if (errors) {
      return {
        errors
      };
    }
    return monitor;
  }
});
exports.runOnceSyntheticsMonitorRoute = runOnceSyntheticsMonitorRoute;