"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.significantEventsRoutes = void 0;
var _zod = require("@kbn/zod");
var _rxjs = require("rxjs");
var _streamlang = require("@kbn/streamlang");
var _zodHelpers = require("@kbn/zod-helpers");
var _constants = require("../../../../common/constants");
var _generate_significant_events = require("../../../lib/significant_events/generate_significant_events");
var _preview_significant_events = require("../../../lib/significant_events/preview_significant_events");
var _read_significant_events_from_alerts_indices = require("../../../lib/significant_events/read_significant_events_from_alerts_indices");
var _create_server_route = require("../../create_server_route");
var _assert_significant_events_access = require("../../utils/assert_significant_events_access");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Make sure strings are expected for input, but still converted to a
// Date, without breaking the OpenAPI generator
const dateFromString = _zod.z.string().transform(input => new Date(input));
const previewSignificantEventsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/significant_events/_preview 2023-10-31',
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      from: dateFromString,
      to: dateFromString,
      bucketSize: _zod.z.string()
    }),
    body: _zod.z.object({
      query: _zod.z.object({
        feature: _zod.z.object({
          name: _zodHelpers.NonEmptyString,
          filter: _streamlang.conditionSchema
        }).optional(),
        kql: _zod.z.object({
          query: _zod.z.string()
        })
      })
    })
  }),
  options: {
    access: 'public',
    summary: 'Preview significant events',
    description: 'Preview significant event results based on a given query',
    availability: {
      since: '9.1.0',
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      streamsClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      body: {
        query
      },
      path: {
        name
      },
      query: {
        bucketSize,
        from,
        to
      }
    } = params;
    const definition = await streamsClient.getStream(name);
    return await (0, _preview_significant_events.previewSignificantEvents)({
      definition,
      bucketSize,
      from,
      to,
      query
    }, {
      scopedClusterClient
    });
  }
});
const readSignificantEventsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/significant_events 2023-10-31',
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      from: dateFromString,
      to: dateFromString,
      bucketSize: _zod.z.string()
    })
  }),
  options: {
    access: 'public',
    summary: 'Read the significant events',
    description: 'Read the significant events',
    availability: {
      since: '9.1.0',
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      streamsClient,
      assetClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    await streamsClient.ensureStream(params.path.name);
    const {
      name
    } = params.path;
    const {
      from,
      to,
      bucketSize
    } = params.query;
    return await (0, _read_significant_events_from_alerts_indices.readSignificantEventsFromAlertsIndices)({
      name,
      from,
      to,
      bucketSize
    }, {
      assetClient,
      scopedClusterClient
    });
  }
});
const generateSignificantEventsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/significant_events/_generate 2023-10-31',
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      connectorId: _zod.z.string(),
      currentDate: dateFromString.optional(),
      from: dateFromString,
      to: dateFromString
    }),
    body: _zod.z.object({
      feature: _zod.z.object({
        name: _zodHelpers.NonEmptyString,
        filter: _streamlang.conditionSchema,
        description: _zod.z.string()
      }).optional()
    })
  }),
  options: {
    access: 'public',
    summary: 'Generate significant events',
    description: 'Generate significant events queries based on the stream data',
    availability: {
      since: '9.2.0',
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    params,
    request,
    getScopedClients,
    server,
    logger
  }) => {
    var _params$body;
    const {
      streamsClient,
      scopedClusterClient,
      licensing,
      inferenceClient,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    await streamsClient.ensureStream(params.path.name);
    const definition = await streamsClient.getStream(params.path.name);
    return (0, _rxjs.from)((0, _generate_significant_events.generateSignificantEventDefinitions)({
      definition,
      feature: (_params$body = params.body) === null || _params$body === void 0 ? void 0 : _params$body.feature,
      connectorId: params.query.connectorId,
      start: params.query.from.valueOf(),
      end: params.query.to.valueOf()
    }, {
      inferenceClient,
      esClient: scopedClusterClient.asCurrentUser,
      logger
    })).pipe((0, _rxjs.mergeMap)(queries => (0, _rxjs.from)(queries)), (0, _rxjs.map)(query => ({
      query,
      type: 'generated_query'
    })));
  }
});
const significantEventsRoutes = exports.significantEventsRoutes = {
  ...readSignificantEventsRoute,
  ...previewSignificantEventsRoute,
  ...generateSignificantEventsRoute
};