"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addAliasesForNamespacedFields = addAliasesForNamespacedFields;
exports.otelEquivalentLookupMap = exports.logsSettings = exports.baseMappings = exports.baseFields = void 0;
var _ecs = require("@elastic/ecs");
var _streamsSchema = require("@kbn/streams-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// This map is used to find the ECS equivalent field for a given OpenTelemetry attribute.
const otelEquivalentLookupMap = exports.otelEquivalentLookupMap = Object.fromEntries(Object.entries(_ecs.EcsFlat).flatMap(([fieldName, field]) => {
  if (!('otel' in field) || !field.otel) {
    return [];
  }
  const otelEquivalentProperty = field.otel.find(otelProperty => otelProperty.relation === 'equivalent');
  if (!otelEquivalentProperty || !('attribute' in otelEquivalentProperty) || !(typeof otelEquivalentProperty.attribute === 'string')) {
    return [];
  }
  return [[otelEquivalentProperty.attribute, fieldName]];
}));
const logsSettings = exports.logsSettings = {
  index: {
    mode: 'logsdb',
    codec: 'best_compression',
    sort: {
      field: ['resource.attributes.host.name', '@timestamp'],
      order: ['asc', 'desc']
    },
    mapping: {
      total_fields: {
        ignore_dynamic_beyond_limit: true
      },
      ignore_malformed: true
    }
  }
};
const baseFields = exports.baseFields = {
  '@timestamp': {
    type: 'date'
  },
  'stream.name': {
    type: 'system'
  },
  'scope.name': {
    type: 'keyword'
  },
  trace_id: {
    type: 'keyword'
  },
  span_id: {
    type: 'keyword'
  },
  event_name: {
    type: 'keyword'
  },
  severity_text: {
    type: 'keyword'
  },
  'body.text': {
    type: 'match_only_text'
  },
  severity_number: {
    type: 'long'
  },
  'resource.attributes.host.name': {
    type: 'keyword'
  },
  'resource.attributes.service.name': {
    type: 'keyword'
  }
};
const baseMappings = exports.baseMappings = {
  body: {
    type: 'object',
    properties: {
      structured: {
        type: 'object',
        subobjects: false
      }
    }
  },
  attributes: {
    type: 'object',
    subobjects: false
  },
  resource: {
    type: 'object',
    properties: {
      attributes: {
        type: 'object',
        subobjects: false
      }
    }
  },
  scope: {
    type: 'object',
    properties: {
      attributes: {
        type: 'object',
        subobjects: false
      }
    }
  },
  'span.id': {
    path: 'span_id',
    type: 'alias'
  },
  message: {
    path: 'body.text',
    type: 'alias'
  },
  'trace.id': {
    path: 'trace_id',
    type: 'alias'
  },
  'log.level': {
    path: 'severity_text',
    type: 'alias'
  }
};

/**
 * Takes a map of fields and returns a sorted array of field names.
 * The sorting sorts fields alphabetically, but puts fields with otelPrefixes at the end in the order of the
 * prefixes array.
 */
function getSortedFields(fields) {
  return Object.entries(fields).sort(([a], [b]) => {
    const aPrefixIndex = _streamsSchema.namespacePrefixes.findIndex(prefix => a.startsWith(prefix));
    const bPrefixIndex = _streamsSchema.namespacePrefixes.findIndex(prefix => b.startsWith(prefix));
    if (aPrefixIndex !== -1 && bPrefixIndex === -1) {
      return 1;
    }
    if (aPrefixIndex === -1 && bPrefixIndex !== -1) {
      return -1;
    }
    if (aPrefixIndex !== -1 && bPrefixIndex !== -1) {
      return aPrefixIndex - bPrefixIndex;
    }
    return a.localeCompare(b);
  });
}
const allNamespacesRegex = new RegExp(`^(${_streamsSchema.namespacePrefixes.join('|')})`);

/**
 * Helper function that creates aliases for fields with namespace prefixes
 * @param fields - The fields to process
 * @param fromSource - The source to set in the 'from' property of the alias
 * @param targetCollection - Where to add the new aliases
 */
const createAliasesForNamespacedFields = (fields, fromSource, targetCollection) => {
  getSortedFields(fields).forEach(([key, fieldDef]) => {
    if (_streamsSchema.namespacePrefixes.some(prefix => key.startsWith(prefix))) {
      const aliasKey = key.replace(allNamespacesRegex, '');
      const from = typeof fromSource === 'function' ? fromSource(key) : fromSource;
      targetCollection[aliasKey] = {
        ...fieldDef,
        from,
        alias_for: key
      };
    }
  });
  // check whether the field has an otel equivalent. If yes, set the ECS equivalent as an alias
  // This needs to be done after the initial properties are set, so the ECS equivalent aliases win out
  getSortedFields(fields).forEach(([key, fieldDef]) => {
    if (_streamsSchema.namespacePrefixes.some(prefix => key.startsWith(prefix))) {
      const aliasKey = key.replace(allNamespacesRegex, '');
      const from = typeof fromSource === 'function' ? fromSource(key) : fromSource;
      const otelEquivalent = otelEquivalentLookupMap[aliasKey];
      if (otelEquivalent) {
        targetCollection[otelEquivalent] = {
          ...fieldDef,
          from,
          alias_for: key
        };
      }
    }
  });
};
function addAliasesForNamespacedFields(streamDefinition, inheritedFields) {
  // Create aliases for inherited fields
  createAliasesForNamespacedFields(inheritedFields, key => inheritedFields[key].from, inheritedFields);

  // Create aliases for this stream's fields
  createAliasesForNamespacedFields(streamDefinition.ingest.wired.fields, streamDefinition.name, inheritedFields);

  // Add aliases defined in the base mappings
  Object.entries(baseMappings).forEach(([key, fieldDef]) => {
    if (fieldDef.type === 'alias') {
      inheritedFields[key] = {
        type: baseFields[fieldDef.path].type,
        alias_for: fieldDef.path,
        from: 'logs'
      };
    }
  });
  return inheritedFields;
}