"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateLayer = generateLayer;
var _streamsSchema = require("@kbn/streams-schema");
var _constants = require("../../../../common/constants");
var _logs_layer = require("./logs_layer");
var _name = require("./name");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function generateLayer(name, definition, isServerless) {
  const properties = {};
  const aliases = {};
  Object.entries(definition.ingest.wired.fields).forEach(([field, props]) => {
    if (props.type === 'system') {
      return;
    }
    const property = {
      type: props.type
    };
    const advancedParameters = (0, _streamsSchema.getAdvancedParameters)(field, props);
    if (Object.keys(advancedParameters).length > 0) {
      Object.assign(property, advancedParameters);
    }
    if (field === '@timestamp') {
      // @timestamp can't ignore malformed dates as it's used for sorting in logsdb
      property.ignore_malformed = false;
    }
    if (props.type === 'date' && props.format) {
      property.format = props.format;
    }
    properties[field] = property;
    const matchingPrefix = _streamsSchema.namespacePrefixes.find(prefix => field.startsWith(prefix));
    if (matchingPrefix) {
      const aliasName = field.substring(matchingPrefix.length);
      aliases[aliasName] = {
        type: 'alias',
        path: field
      };
    }
  });

  // check whether the field has an otel equivalent. If yes, set the ECS equivalent as an alias
  // This needs to be done after the initial properties are set, so the ECS equivalent aliases win out
  Object.entries(definition.ingest.wired.fields).forEach(([field, props]) => {
    const matchingPrefix = _streamsSchema.namespacePrefixes.find(prefix => field.startsWith(prefix));
    if (matchingPrefix) {
      const aliasName = field.substring(matchingPrefix.length);
      const otelEquivalent = _logs_layer.otelEquivalentLookupMap[aliasName];
      if (otelEquivalent) {
        aliases[otelEquivalent] = {
          type: 'alias',
          path: field
        };
      }
    }
  });
  return {
    name: (0, _name.getComponentTemplateName)(name),
    template: {
      settings: getTemplateSettings(definition, isServerless),
      mappings: {
        dynamic: false,
        properties: (0, _streamsSchema.isRoot)(name) ? {
          ..._logs_layer.baseMappings,
          ...properties,
          ...aliases
        } : {
          ...properties,
          ...aliases
        }
      }
    },
    version: _constants.ASSET_VERSION,
    _meta: {
      managed: true,
      description: `Default settings for the ${name} stream`
    }
  };
}
function getTemplateSettings(definition, isServerless) {
  const baseSettings = (0, _streamsSchema.isRoot)(definition.name) ? _logs_layer.logsSettings : {};
  return baseSettings;
}