"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EMPTY_EQUALS_CONDITION = void 0;
exports.alwaysToEmptyEquals = alwaysToEmptyEquals;
exports.conditionNeedsValueField = conditionNeedsValueField;
exports.emptyEqualsToAlways = emptyEqualsToAlways;
exports.isConditionEditableInUi = void 0;
exports.isShorthandBooleanFilterCondition = isShorthandBooleanFilterCondition;
var _streamlang = require("@kbn/streamlang");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EMPTY_EQUALS_CONDITION = exports.EMPTY_EQUALS_CONDITION = Object.freeze({
  field: '',
  eq: ''
});
function alwaysToEmptyEquals(condition) {
  if ((0, _streamlang.isAlwaysCondition)(condition)) {
    return (0, _lodash.cloneDeep)(EMPTY_EQUALS_CONDITION);
  }
  return condition;
}
function emptyEqualsToAlways(condition) {
  if ((0, _lodash.isEqual)(condition, EMPTY_EQUALS_CONDITION)) {
    return _streamlang.ALWAYS_CONDITION;
  }
  return condition;
}
const UI_SUPPORTED_OPERATORS_AND_VALUE_TYPES = {
  // Allow both string and boolean for eq/neq so that boolean shorthand (e.g. "equals true") can rendered in UI
  eq: ['string', 'boolean'],
  neq: ['string', 'boolean'],
  gt: ['string'],
  gte: ['string'],
  lt: ['string'],
  lte: ['string'],
  contains: ['string'],
  startsWith: ['string'],
  endsWith: ['string'],
  exists: ['boolean']
};
function isOperatorUiSupported(operator) {
  return operator in UI_SUPPORTED_OPERATORS_AND_VALUE_TYPES;
}
const isConditionEditableInUi = condition => {
  if ((0, _lodash.isPlainObject)(condition) && (0, _streamlang.isFilterConditionObject)(condition)) {
    const operator = (0, _streamlang.getFilterOperator)(condition);
    const value = (0, _streamlang.getFilterValue)(condition);

    // Check if the operator itself is supported by the UI
    if (!operator || !isOperatorUiSupported(operator)) {
      return false;
    }

    // Check if the value's data type is supported for that specific operator
    const allowedTypes = UI_SUPPORTED_OPERATORS_AND_VALUE_TYPES[operator];
    if (!allowedTypes) {
      return false;
    }
    return allowedTypes.includes(typeof value);
  }

  // If it's not a simple filter, the only other UI-representable state is
  // an 'always' condition, which is representable in UI (empty condition)
  return (0, _streamlang.isAlwaysCondition)(condition);
};

// Determines whether a filter condition can be represented using the boolean shorthand
// e.g. { field: 'foo', eq: true } can be represented by "equals true" operator in the UI
exports.isConditionEditableInUi = isConditionEditableInUi;
function isShorthandBooleanFilterCondition(condition) {
  return (0, _lodash.isPlainObject)(condition) && (0, _streamlang.isFilterConditionObject)(condition) && 'eq' in condition && typeof condition.eq === 'boolean' || 'neq' in condition && typeof condition.neq === 'boolean';
}

// Determines whether a value input should be displayed for a condition.
// Shorthand binary operators e.g. "equals true" do not need a value field.
function conditionNeedsValueField(condition) {
  return !isShorthandBooleanFilterCondition(condition);
}